@extends('layouts/layoutMaster')

@section('title', 'Rentals - Equipment Renting System')

<!-- Vendor Styles -->
@section('vendor-style')
  @vite(['resources/assets/vendor/libs/datatables-bs5/datatables.bootstrap5.scss', 'resources/assets/vendor/libs/datatables-responsive-bs5/responsive.bootstrap5.scss', 'resources/assets/vendor/libs/datatables-buttons-bs5/buttons.bootstrap5.scss', 'resources/assets/vendor/libs/flatpickr/flatpickr.scss', 'resources/assets/vendor/libs/@form-validation/form-validation.scss', 'resources/assets/vendor/libs/select2/select2.scss'])
@endsection

<!-- Vendor Scripts -->
@section('vendor-script')
  @vite(['resources/assets/vendor/libs/datatables-bs5/datatables-bootstrap5.js', 'resources/assets/vendor/libs/moment/moment.js', 'resources/assets/vendor/libs/flatpickr/flatpickr.js', 'resources/assets/vendor/libs/@form-validation/popular.js', 'resources/assets/vendor/libs/@form-validation/bootstrap5.js', 'resources/assets/vendor/libs/@form-validation/auto-focus.js', 'resources/assets/vendor/libs/select2/select2.js'])
@endsection

@section('page-style')
  <style>
    .select2-container {
      z-index: 9999 !important;
    }

    .select2-dropdown {
      z-index: 9999 !important;
    }
  </style>
@endsection

@section('content')
  <meta name="csrf-token" content="{{ csrf_token() }}">
  <!-- DataTable with Buttons -->
  <div class="card">
    <div class="card-datatable text-nowrap">
      <table class="datatables-basic table table-bordered table-responsive">
        <thead>
          <tr>
            <th></th>
            <th></th>
            <th>ID</th>
            <th>Customer</th>
            <th>Shop</th>
            <th>Rental Date</th>
            <th>Expected Return</th>
            <th>Status</th>
            <th>Total Amount</th>
            <th class="d-flex align-items-center">Action</th>
          </tr>
        </thead>
        <tbody>
          @foreach ($rentals as $rental)
            <tr data-rental-id="{{ $rental->id }}">
              <td></td>
              <td><input type="checkbox" class="dt-checkboxes form-check-input"></td>
              <td>{{ $loop->iteration + ($rentals->currentPage() - 1) * $rentals->perPage() }}</td>
              <td>
                <div class="d-flex justify-content-start align-items-center user-name">
                  <div class="avatar-wrapper">
                    <div class="avatar me-2">
                      <span class="avatar-initial rounded-circle bg-label-primary">
                        {{ strtoupper(substr($rental->customer?->name ?? 'N/A', 0, 2)) }}
                      </span>
                    </div>
                  </div>
                  <div class="d-flex flex-column">
                    <span class="customer_name text-truncate h6 mb-0">{{ $rental->customer?->name ?? 'N/A' }}</span>
                    <small class="text-truncate">{{ $rental->customer?->phone ?? '' }}</small>
                  </div>
                </div>
              </td>
              <td>
                <div class="d-flex flex-column">
                  <span class="text-truncate h6 mb-0">{{ $rental->shop->name ?? 'N/A' }}</span>
                  <small class="text-truncate">{{ $rental->invoices->first()->invoice_number ?? '' }}</small>
                  <small class="text-truncate">{{ $rental->agreement->agreement_number ?? '' }}</small>
                </div>
              </td>
              <td>{{ $rental->rental_date->format('Y-m-d H:i') }}</td>
              <td>{{ $rental->expected_return_date ? $rental->expected_return_date->format('Y-m-d H:i') : 'N/A' }}</td>
              <td>
                @php
                  $statusColors = [
                      'draft' => 'bg-label-secondary',
                      'active' => 'bg-label-success',
                      'returned' => 'bg-label-info',
                      'cancelled' => 'bg-label-danger',
                  ];
                  $color = $statusColors[$rental->status] ?? 'bg-label-secondary';
                @endphp
                <span class="badge {{ $color }}">{{ ucfirst($rental->status) }}</span>
              </td>
              <td>{{ number_format($rental->grand_total, 2) }}</td>
              <td>
                <div class="d-inline-block">
                  <a href="javascript:;"
                    class="btn btn-sm btn-text-secondary rounded-pill btn-icon dropdown-toggle hide-arrow"
                    data-bs-toggle="dropdown">
                    <i class="icon-base ri ri-more-2-line icon-20px"></i>
                  </a>
                  <ul class="dropdown-menu dropdown-menu-end m-0">
                    @if($rental->status === 'returned')
                      @php
                        $invoice = $rental->invoices->first();
                      @endphp
                      @if($invoice)
                        <li><a href="{{ route('invoices.preview', $invoice->id) }}" class="dropdown-item item-view">
                            <i class="icon-base ri ri-file-list-3-line me-2"></i>View Invoice</a></li>
                        <div class="dropdown-divider"></div>
                      @else
                        <li><a href="{{ route('invoices.generate', $rental->id) }}" class="dropdown-item item-view">
                            <i class="icon-base ri ri-file-add-line me-2"></i>Generate Invoice</a></li>
                        <div class="dropdown-divider"></div>
                      @endif
                    @endif
                    <li><a href="{{ route('rentals.view', $rental->id) }}" class="dropdown-item item-view">
                        <i class="icon-base ri ri-file-text-line me-2"></i>View Agreement (Job #{{ $rental->id }})</a></li>
                    <li><a href="javascript:;" class="dropdown-item item-details">Details</a></li>
                    <div class="dropdown-divider"></div>
                    <li><a href="javascript:;" class="dropdown-item text-danger delete-record"
                        data-id="{{ $rental->id }}">Delete</a></li>
                  </ul>
                </div>
                <a href="javascript:;" class="btn btn-sm btn-text-secondary rounded-pill btn-icon item-edit">
                  <i class="icon-base ri ri-edit-box-line icon-20px"></i>
                </a>
              </td>
            </tr>
          @endforeach
        </tbody>
      </table>
    </div>
  </div>

  <!-- Modal to add/edit rental record -->
  <div class="offcanvas offcanvas-end" id="add-new-record">
    <div class="offcanvas-header border-bottom">
      <h5 class="offcanvas-title" id="formTitle">New Rental</h5>
      <button type="button" class="btn-close text-reset" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body flex-grow-1">
      <form method="POST" class="add-new-record pt-0 row g-3" id="rentalForm">
        @csrf
        <input type="hidden" name="_method" id="methodField" value="POST">
        <input type="hidden" id="rentalId" name="rentalId" />

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="basicShop2" class="input-group-text">
              <i class="icon-base ri ri-store-2-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              @php
                $user = auth()->user();
                $isAdmin = $user && $user->isAdmin();
                $userShopId = $user && !$isAdmin ? $user->shop_id : null;
              @endphp
              <select id="basicShop" name="basicShop" class="select2 form-select shop dt-shop" placeholder="Shop"
                aria-label="Shop" aria-describedby="basicShop2" @if(!$isAdmin) disabled @endif>
                <option value=""></option>
                @foreach (\App\Models\Shop::all() as $shop)
                  <option value="{{ $shop->id }}" @if($userShopId && $shop->id == $userShopId) selected @endif>{{ $shop->name }}</option>
                @endforeach
              </select>
              <label for="basicShop" class="form-label">Shop</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="basicCustomer2" class="input-group-text">
              <i class="icon-base ri ri-user-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <select id="basicCustomer" name="basicCustomer" class="select2 form-select customer dt-customer"
                placeholder="Customer" aria-label="Customer" aria-describedby="basicCustomer2" data-allow-clear="true">
                <option value=""></option>
                @foreach (\App\Models\Customer::all() as $customer)
                  <option value="{{ $customer->id }}">{{ $customer->name }} - {{ $customer->phone }}</option>
                @endforeach
              </select>
              <label for="basicCustomer" class="form-label">Customer (Optional)</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="basicStatus2" class="input-group-text">
              <i class="icon-base ri ri-checkbox-circle-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <select id="basicStatus" name="basicStatus" class="form-select dt-status" aria-label="Status"
                aria-describedby="basicStatus2">
                <option value="draft">Draft</option>
                <option value="active">Active</option>
                <option value="returned">Returned</option>
                <option value="cancelled">Cancelled</option>
              </select>
              <label for="basicStatus">Status</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="basicRentalDate2" class="input-group-text">
              <i class="icon-base ri ri-calendar-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <input type="datetime-local" id="basicRentalDate" name="basicRentalDate"
                class="form-control dt-rental-date" aria-label="Rental Date" aria-describedby="basicRentalDate2" />
              <label for="basicRentalDate">Rental Date</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="basicExpectedReturn2" class="input-group-text">
              <i class="icon-base ri ri-calendar-check-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <input type="datetime-local" id="basicExpectedReturn" name="basicExpectedReturn"
                class="form-control dt-expected-return" aria-label="Expected Return Date"
                aria-describedby="basicExpectedReturn2" />
              <label for="basicExpectedReturn">Expected Return Date (Optional)</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="basicRemarks2" class="input-group-text">
              <i class="icon-base ri ri-file-text-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <textarea class="form-control dt-remarks" id="basicRemarks" name="basicRemarks" placeholder="Remarks"
                aria-label="Remarks" aria-describedby="basicRemarks2" rows="3"></textarea>
              <label for="basicRemarks">Remarks (Optional)</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="basicGuaranteeType2" class="input-group-text">
              <i class="icon-base ri ri-shield-check-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <select id="basicGuaranteeType" name="basicGuaranteeType" class="form-select dt-guarantee-type" aria-label="Guarantee Type"
                aria-describedby="basicGuaranteeType2">
                <option value="">Select Guarantee (Optional)</option>
                <option value="cash">Cash</option>
                <option value="nic">NIC</option>
                <option value="driving_license">Driving License</option>
                <option value="other">Other</option>
              </select>
              <label for="basicGuaranteeType">Guarantee Type</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation" id="guaranteeDetailsWrapper" style="display: none;">
          <div class="input-group input-group-merge">
            <span id="basicGuaranteeDetails2" class="input-group-text">
              <i class="icon-base ri ri-file-text-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <textarea class="form-control dt-guarantee-details" id="basicGuaranteeDetails" name="basicGuaranteeDetails" placeholder="Guarantee Details"
                aria-label="Guarantee Details" aria-describedby="basicGuaranteeDetails2" rows="3"></textarea>
              <label for="basicGuaranteeDetails">Guarantee Details</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12">
          <button type="submit" id="submitBtn" class="btn btn-primary data-submit me-sm-4 me-1">Submit</button>
          <button type="reset" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">Cancel</button>
        </div>
      </form>
    </div>
  </div>

@endsection

@section('page-script')
  <script>
    'use strict';

      let fv, offCanvasEl, formAddNewRecord;
      document.addEventListener('DOMContentLoaded', function(e) {
        (function() {
          formAddNewRecord = document.getElementById('rentalForm');
          var form = document.getElementById('rentalForm');

        // Clean up any duplicate Select2 containers that might exist
        $('.select2-container').each(function() {
          const $container = $(this);
          const $prev = $container.prev();
          // Remove if there's no corresponding select element
          if ($prev.length === 0 || !$prev.is('select')) {
            $container.remove();
          }
        });

        form.addEventListener('submit', function(e) {
          e.preventDefault();
        });

        setTimeout(() => {
          const newRecord = document.querySelector('.create-new-form-btn'),
            offCanvasElement = document.querySelector('#add-new-record');

          if (newRecord) {
            newRecord.addEventListener('click', function() {
              offCanvasEl = new bootstrap.Offcanvas(offCanvasElement);
              document.getElementById('formTitle').textContent = 'New Rental';
              document.getElementById('rentalId').value = '';
              formAddNewRecord.reset();
              // Auto-select shop for non-admin users
              @php
                $user = auth()->user();
                $isAdmin = $user && $user->isAdmin();
                $userShopId = $user && !$isAdmin ? $user->shop_id : null;
              @endphp
              @if($userShopId)
              const shopSelect = document.querySelector('.dt-shop');
              if (shopSelect) {
                shopSelect.value = '{{ $userShopId }}';
                // Ensure it's disabled for non-admin users
                shopSelect.disabled = true;
                $(shopSelect).prop('disabled', true);
                // Update Select2 to reflect disabled state and trigger change
                if ($(shopSelect).hasClass('select2-hidden-accessible')) {
                  $(shopSelect).select2('enable', false);
                  $(shopSelect).trigger('change.select2');
                } else {
                  $(shopSelect).trigger('change');
                }
              }
              @endif
              // Set default rental date to now
              const now = new Date();
              now.setMinutes(now.getMinutes() - now.getTimezoneOffset());
              document.querySelector('.dt-rental-date').value = now.toISOString().slice(0, 16);
              // Set default rented_by to current user
              offCanvasEl.show();
            });
          }
        }, 200);

        fv = FormValidation.formValidation(formAddNewRecord, {
          fields: {
            basicShop: {
              validators: {
                notEmpty: {
                  message: 'Shop is required'
                }
              }
            },
            basicRentalDate: {
              validators: {
                notEmpty: {
                  message: 'Rental date is required'
                }
              }
            }
          },
          plugins: {
            trigger: new FormValidation.plugins.Trigger(),
            bootstrap5: new FormValidation.plugins.Bootstrap5({
              eleValidClass: '',
              rowSelector: '.form-control-validation'
            }),
            submitButton: new FormValidation.plugins.SubmitButton(),
            autoFocus: new FormValidation.plugins.AutoFocus()
          },
          init: instance => {
            instance.on('plugins.message.placed', function(e) {
              if (e.element.parentElement.classList.contains('input-group')) {
                e.element.parentElement.insertAdjacentElement('afterend', e.messageElement);
              }
            });
          }
        });
      })();

      const dt_basic_table = document.querySelector('.datatables-basic');
      let dt_basic;

      if (dt_basic_table) {
        let tableTitle = document.createElement('h5');
        tableTitle.classList.add('card-title', 'mb-0', 'text-md-start', 'text-center');
        tableTitle.innerHTML = 'Rentals';

        dt_basic = new DataTable(dt_basic_table, {
          columns: [{
              data: 'id'
            },
            {
              data: 'id',
              orderable: false,
              render: DataTable.render.select()
            },
            {
              data: 'id'
            },
            {
              data: 'customer'
            },
            {
              data: 'shop'
            },
            {
              data: 'rental_date'
            },
            {
              data: 'expected_return'
            },
            {
              data: 'status'
            },
            {
              data: 'total'
            },
            {
              data: 'id'
            }
          ],
          columnDefs: [{
              className: 'control',
              orderable: false,
              searchable: false,
              responsivePriority: 2,
              targets: 0,
              render: function(data, type, full, meta) {
                return '';
              }
            },
            {
              targets: 1,
              orderable: false,
              searchable: false,
              responsivePriority: 3,
              checkboxes: true,
              render: function() {
                return '<input type="checkbox" class="dt-checkboxes form-check-input">';
              }
            },
            {
              targets: 2,
              searchable: false,
              visible: false
            },
            {
              responsivePriority: 1,
              targets: 3
            },
            {
              targets: -1,
              title: 'Actions',
              orderable: false,
              searchable: false,
              className: 'd-flex align-items-center',
              render: function(data, type, full, meta) {
                return data;
              }
            }
          ],
          select: {
            style: 'multi',
            selector: 'td:nth-child(2)'
          },
          order: [
            [2, 'desc']
          ],
          layout: {
            top2Start: {
              rowClass: 'row card-header mx-0 px-2',
              features: [tableTitle]
            },
            top2End: {
              features: [{
                buttons: [{
                    extend: 'collection',
                    className: 'btn btn-label-primary dropdown-toggle me-4 waves-effect border-none',
                    text: '<span class="d-flex align-items-center gap-2"><i class="icon-base ri ri-external-link-line icon-18px"></i> <span class="d-none d-sm-inline-block">Export</span></span>',
                    buttons: [{
                        extend: 'print',
                        text: `<span class="d-flex align-items-center"><i class="icon-base ri ri-printer-line me-1"></i>Print</span>`,
                        className: 'dropdown-item',
                        exportOptions: {
                          columns: [3, 4, 5, 6, 7, 8]
                        }
                      },
                      {
                        extend: 'csv',
                        text: `<span class="d-flex align-items-center"><i class="icon-base ri ri-file-text-line me-1"></i>Csv</span>`,
                        className: 'dropdown-item',
                        exportOptions: {
                          columns: [3, 4, 5, 6, 7, 8]
                        }
                      },
                      {
                        extend: 'excel',
                        text: `<span class="d-flex align-items-center"><i class="icon-base ri ri-file-excel-line me-1"></i>Excel</span>`,
                        className: 'dropdown-item',
                        exportOptions: {
                          columns: [3, 4, 5, 6, 7, 8]
                        }
                      },
                      {
                        extend: 'pdf',
                        text: `<span class="d-flex align-items-center"><i class="icon-base ri ri-file-pdf-line me-1"></i>Pdf</span>`,
                        className: 'dropdown-item',
                        exportOptions: {
                          columns: [3, 4, 5, 6, 7, 8]
                        }
                      },
                      {
                        extend: 'copy',
                        text: `<i class="icon-base ri ri-file-copy-line me-1"></i>Copy`,
                        className: 'dropdown-item',
                        exportOptions: {
                          columns: [3, 4, 5, 6, 7, 8]
                        }
                      }
                    ]
                  },
                  {
                    text: '<span class="d-flex align-items-center"><i class="icon-base ri ri-add-line icon-18px me-sm-1"></i><span class="d-none d-sm-inline-block">Add New Record</span></span>',
                    className: 'btn btn-primary',
                    action: function() {
                      window.location.href = '{{ route('rentals.create') }}';
                    }
                  }
                ]
              }]
            },
            topStart: {
              rowClass: 'row m-3 mx-2 my-0 justify-content-between',
              features: [{
                pageLength: {
                  menu: [7, 10, 25, 50, 100],
                  text: 'Show_MENU_entries'
                }
              }]
            },
            topEnd: {
              search: {
                placeholder: 'Type search here'
              }
            },
            bottomStart: {
              rowClass: 'row mx-3 justify-content-between',
              features: ['info']
            },
            bottomEnd: 'paging'
          },
          displayLength: 7,
          language: {
            paginate: {
              next: '<i class="icon-base ri ri-arrow-right-s-line scaleX-n1-rtl icon-22px"></i>',
              previous: '<i class="icon-base ri ri-arrow-left-s-line scaleX-n1-rtl icon-22px"></i>',
              first: '<i class="icon-base ri ri-skip-back-mini-line scaleX-n1-rtl icon-22px"></i>',
              last: '<i class="icon-base ri ri-skip-forward-mini-line scaleX-n1-rtl icon-22px"></i>'
            }
          },
          responsive: {
            details: {
              display: DataTable.Responsive.display.modal({
                header: function(row) {
                  const data = row.data();
                  return 'Details of Rental';
                }
              }),
              type: 'column',
              renderer: function(api, rowIdx, columns) {
                const data = columns
                  .map(function(col) {
                    return col.title !== '' ?
                      `<tr data-dt-row="${col.rowIndex}" data-dt-column="${col.columnIndex}">
                      <td>${col.title}:</td>
                      <td>${col.data}</td>
                    </tr>` :
                      '';
                  })
                  .join('');

                if (data) {
                  const div = document.createElement('div');
                  div.classList.add('table-responsive');
                  const table = document.createElement('table');
                  div.appendChild(table);
                  table.classList.add('table');
                  table.classList.add('datatables-basic');
                  const tbody = document.createElement('tbody');
                  tbody.innerHTML = data;
                  table.appendChild(tbody);
                  return div;
                }
                return false;
              }
            }
          },
          initComplete: function(settings, json) {
            $('.card-header').after('<hr class="my-0">');
          }
        });

        fv.on('core.form.valid', function() {
          let rentalId = document.querySelector('#rentalId').value;
          let new_shop = document.querySelector('.add-new-record .dt-shop').value;
          let new_customer = document.querySelector('.add-new-record .dt-customer').value;
          let new_status = document.querySelector('.add-new-record .dt-status').value;
          let new_rental_date = document.querySelector('.add-new-record .dt-rental-date').value;
          let new_expected_return = document.querySelector('.add-new-record .dt-expected-return').value;
          let new_remarks = document.querySelector('.add-new-record .dt-remarks').value;
          let new_guarantee_type = document.querySelector('.add-new-record .dt-guarantee-type').value;
          let new_guarantee_details = document.querySelector('.add-new-record .dt-guarantee-details').value;

          const isEditing = rentalId && rentalId.trim() !== '';
          const url = isEditing ? `/api/rentals/${rentalId}` : '/api/rentals';
          const method = isEditing ? 'PUT' : 'POST';

          fetch(url, {
              method: method,
              headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
              },
              credentials: 'include',
              body: JSON.stringify({
                shop_id: parseInt(new_shop),
                customer_id: new_customer ? parseInt(new_customer) : null,
                rented_by: isEditing ? undefined : {{ auth()->id() ?? 1 }},
                status: new_status,
                rental_date: new_rental_date,
                expected_return_date: new_expected_return || null,
                remarks: new_remarks || null,
                guarantee_type: new_guarantee_type || null,
                guarantee_details: new_guarantee_type === 'other' ? (new_guarantee_details || null) : null
              })
            })
            .then(response => {
              if (!response.ok) {
                return response.json().then(err => {
                  throw new Error(err.message || 'Request failed');
                });
              }
              return response.json();
            })
            .then(data => {
              if (data.success) {
                // Close the offcanvas
                if (offCanvasEl) {
                  offCanvasEl.hide();
                }
                // Reset form
                formAddNewRecord.reset();
                // Reload the page to refresh the table
                window.location.reload();
              } else {
                alert(data.message || 'An error occurred');
              }
            })
            .catch(error => {
              console.error('Error:', error);
              alert('An error occurred while saving the rental: ' + error.message);
            });
          return false;
        });

        document.addEventListener('click', function(e) {
          if (e.target.closest('.item-edit')) {
            e.preventDefault();
            const row = e.target.closest('tr');
            const rentalId = row.getAttribute('data-rental-id');

            // Fetch rental data from API
            fetch(`/api/rentals/${rentalId}`, {
              method: 'GET',
              headers: {
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
              },
              credentials: 'include'
            })
              .then(response => response.json())
              .then(data => {
                if (data.success && data.data) {
                  const rental = data.data;

                  // Populate form fields
                  document.getElementById('formTitle').textContent = 'Edit Rental';
                  document.getElementById('rentalId').value = rental.id;

                  // Set shop
                  const shopSelect = document.querySelector('.dt-shop');
                  if (shopSelect && rental.shop_id) {
                    shopSelect.value = rental.shop_id;
                    // Reinitialize Select2 if needed to update the display
                    if ($(shopSelect).hasClass('select2-hidden-accessible')) {
                      $(shopSelect).trigger('change.select2');
                    } else {
                      $(shopSelect).trigger('change');
                    }
                  }
                  // Disable shop selection for non-admin users when editing
                  @php
                    $user = auth()->user();
                    $isAdmin = $user && $user->isAdmin();
                  @endphp
                  @if(!$isAdmin)
                  if (shopSelect) {
                    shopSelect.disabled = true;
                    $(shopSelect).prop('disabled', true);
                    // Update Select2 to reflect disabled state
                    if ($(shopSelect).hasClass('select2-hidden-accessible')) {
                      $(shopSelect).select2('enable', false);
                    }
                  }
                  @endif

                  // Set customer
                  const customerSelect = document.querySelector('.dt-customer');
                  if (customerSelect && rental.customer_id) {
                    customerSelect.value = rental.customer_id;
                    // Update Select2 display
                    if ($(customerSelect).hasClass('select2-hidden-accessible')) {
                      $(customerSelect).trigger('change.select2');
                    } else {
                      $(customerSelect).trigger('change');
                    }
                  } else if (customerSelect) {
                    customerSelect.value = '';
                    if ($(customerSelect).hasClass('select2-hidden-accessible')) {
                      $(customerSelect).trigger('change.select2');
                    } else {
                      $(customerSelect).trigger('change');
                    }
                  }

                  // Set status
                  const statusSelect = document.querySelector('.dt-status');
                  if (statusSelect && rental.status) {
                    statusSelect.value = rental.status;
                  }

                  // Set rental date
                  const rentalDateInput = document.querySelector('.dt-rental-date');
                  if (rentalDateInput && rental.rental_date) {
                    const rentalDate = new Date(rental.rental_date);
                    const formattedDate = rentalDate.toISOString().slice(0, 16);
                    rentalDateInput.value = formattedDate;
                  }

                  // Set expected return date
                  const expectedReturnInput = document.querySelector('.dt-expected-return');
                  if (expectedReturnInput && rental.expected_return_date) {
                    const expectedDate = new Date(rental.expected_return_date);
                    const formattedExpectedDate = expectedDate.toISOString().slice(0, 16);
                    expectedReturnInput.value = formattedExpectedDate;
                  } else if (expectedReturnInput) {
                    expectedReturnInput.value = '';
                  }

                  // Set remarks
                  const remarksTextarea = document.querySelector('.dt-remarks');
                  if (remarksTextarea) {
                    remarksTextarea.value = rental.remarks || '';
                  }

                  // Set guarantee type
                  const guaranteeTypeSelect = document.querySelector('.dt-guarantee-type');
                  if (guaranteeTypeSelect) {
                    guaranteeTypeSelect.value = rental.guarantee_type || '';
                    // Trigger change to show/hide guarantee details
                    guaranteeTypeSelect.dispatchEvent(new Event('change'));
                  }

                  // Set guarantee details
                  const guaranteeDetailsTextarea = document.querySelector('.dt-guarantee-details');
                  if (guaranteeDetailsTextarea) {
                    guaranteeDetailsTextarea.value = rental.guarantee_details || '';
                  }

                  // Open offcanvas
                  offCanvasEl = new bootstrap.Offcanvas(document.querySelector('#add-new-record'));
                  offCanvasEl.show();
                } else {
                  alert('Error loading rental data');
                }
              })
              .catch(error => {
                console.error('Error:', error);
                alert('Error loading rental data');
              });
          }
        });

        document.addEventListener('click', function(e) {
          if (e.target.classList.contains('delete-record')) {
            e.preventDefault();
            const row = e.target.closest('tr');
            const rentalId = row.getAttribute('data-rental-id');

            if (confirm('Are you sure you want to delete this rental?')) {
              fetch(`/api/rentals/${rentalId}`, {
                  method: 'DELETE',
                  headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                  }
                })
                .then(response => response.json())
                .then(data => {
                  if (data.success) {
                    row.remove();
                  }
                })
                .catch(error => console.error('Error:', error));
            }
          }
        });
      }

      setTimeout(() => {
        const elementsToModify = [{
            selector: '.dt-buttons .btn',
            classToRemove: 'btn-secondary'
          },
          {
            selector: '.dt-layout-table',
            classToRemove: 'row mt-2'
          },
          {
            selector: '.dt-layout-end',
            classToAdd: 'mt-0',
            classToRemove: 'ms-auto'
          },
          {
            selector: '.dt-layout-end .dt-search',
            classToAdd: 'mt-md-5 mt-0',
            classToRemove: 'ms-auto'
          },
          {
            selector: '.dt-layout-full',
            classToRemove: 'col-md col-12',
            classToAdd: 'table-responsive'
          }
        ];

        elementsToModify.forEach(({
          selector,
          classToRemove,
          classToAdd
        }) => {
          document.querySelectorAll(selector).forEach(element => {
            if (classToRemove) {
              classToRemove.split(' ').forEach(className => element.classList.remove(className));
            }
            if (classToAdd) {
              classToAdd.split(' ').forEach(className => element.classList.add(className));
            }
          });
        });
      }, 100);

      // Helper function to safely initialize Select2
      function initializeSelect2(selector, options) {
        const $element = $(selector);
        if ($element.length === 0) return;
        
        // Destroy existing Select2 instance if it exists
        if ($element.hasClass('select2-hidden-accessible')) {
          try {
            $element.select2('destroy');
          } catch (e) {
            console.warn('Error destroying Select2:', e);
          }
          // Remove any leftover containers
          $element.next('.select2-container').remove();
          $element.removeClass('select2-hidden-accessible');
          $element.removeAttr('data-select2-id');
          $element.removeAttr('aria-hidden');
          $element.removeAttr('tabindex');
        }
        
        // Remove any duplicate containers
        $('.select2-container').filter(function() {
          return $(this).prev(selector).length > 0;
        }).remove();
        
        // Initialize Select2 with options
        $element.select2(options);
      }

      // Initialize shop dropdown with auto-selection for non-admin users
      @php
        $user = auth()->user();
        $isAdmin = $user && $user->isAdmin();
        $userShopId = $user && !$isAdmin ? $user->shop_id : null;
      @endphp
      
      // Initialize shop dropdown
      initializeSelect2('#basicShop', {
        placeholder: 'Select Shop',
        allowClear: {{ $isAdmin ? 'true' : 'false' }},
        dropdownParent: $('#add-new-record').length ? $('#add-new-record') : $('body')
      });
      
      // Ensure shop dropdown closes after selection
      $('#basicShop').off('select2:select').on('select2:select', function() {
        $(this).trigger('select2:close');
      });
      
      // Initialize customer dropdown
      initializeSelect2('#basicCustomer', {
        placeholder: 'Select Customer',
        allowClear: true,
        dropdownParent: $('#add-new-record').length ? $('#add-new-record') : $('body')
      });
      
      // Ensure customer dropdown closes after selection
      $('#basicCustomer').off('select2:select').on('select2:select', function() {
        $(this).trigger('select2:close');
      });
      
      @if($userShopId)
        // Auto-select user's shop for non-admin users
        $('#basicShop').val('{{ $userShopId }}').trigger('change');
      @endif

      // Handle guarantee type change to show/hide guarantee details
      const guaranteeTypeSelect = document.getElementById('basicGuaranteeType');
      const guaranteeDetailsWrapper = document.getElementById('guaranteeDetailsWrapper');

      if (guaranteeTypeSelect && guaranteeDetailsWrapper) {
        guaranteeTypeSelect.addEventListener('change', function() {
          if (this.value === 'other') {
            guaranteeDetailsWrapper.style.display = 'block';
            document.getElementById('basicGuaranteeDetails').required = true;
          } else {
            guaranteeDetailsWrapper.style.display = 'none';
            document.getElementById('basicGuaranteeDetails').required = false;
            document.getElementById('basicGuaranteeDetails').value = '';
          }
        });
      }
    });
  </script>
@endsection
