@extends('layouts/layoutMaster')

@section('title', 'Create Rental - Equipment Renting System')

<!-- Vendor Styles -->
@section('vendor-style')
  @vite(['resources/assets/vendor/libs/datatables-bs5/datatables.bootstrap5.scss', 'resources/assets/vendor/libs/datatables-responsive-bs5/responsive.bootstrap5.scss', 'resources/assets/vendor/libs/datatables-buttons-bs5/buttons.bootstrap5.scss', 'resources/assets/vendor/libs/flatpickr/flatpickr.scss', 'resources/assets/vendor/libs/@form-validation/form-validation.scss', 'resources/assets/vendor/libs/select2/select2.scss'])
@endsection

@section('page-style')
  @vite('resources/assets/vendor/scss/pages/app-invoice.scss')
@endsection

@section('vendor-script')
  @vite(['resources/assets/vendor/libs/flatpickr/flatpickr.js', 'resources/assets/vendor/libs/cleave-zen/cleave-zen.js', 'resources/assets/vendor/libs/jquery-repeater/jquery-repeater.js', 'resources/assets/vendor/libs/select2/select2.js', 'resources/assets/vendor/libs/@form-validation/popular.js', 'resources/assets/vendor/libs/@form-validation/bootstrap5.js', 'resources/assets/vendor/libs/@form-validation/auto-focus.js'])
@endsection

@section('page-script')
  <script>
    'use strict';
    // Flag to prevent app-invoice-edit.js from initializing repeater - MUST be set before DOMContentLoaded
    window.rentalFormPage = true;

    let rentalItems = [];
    let currentRentalId = null;
    let equipmentData = [];
    let stockData = {};

    let repeaterInitialized = false;
    let equipmentDataLoaded = false;

    // Function to initialize equipment select dropdown
    function initializeEquipmentSelect(itemWrapper) {
      const select = itemWrapper.find('.item-details');

      if (select.length === 0) {
        console.warn('Equipment select not found in item wrapper');
        return;
      }

      // Check if Select2 is already initialized and destroy it properly
      if (select.hasClass('select2-hidden-accessible') || select.next('.select2-container').length > 0) {
        try {
          // Destroy Select2 instance
          select.select2('destroy');
        } catch (e) {
          console.warn('Error destroying Select2:', e);
        }
        // Remove any leftover Select2 containers
        select.next('.select2-container').remove();
        // Remove Select2 classes and attributes
        select.removeClass('select2-hidden-accessible');
        select.removeAttr('data-select2-id');
        select.removeAttr('aria-hidden');
        select.removeAttr('tabindex');
      }

      // Also check for any duplicate Select2 containers in the wrapper and remove them
      // But only remove containers that are NOT next to our select
      itemWrapper.find('.select2-container').each(function() {
        const $container = $(this);
        // Only remove if it's not immediately after our select
        if ($container.prev('.item-details').length === 0) {
          $container.remove();
        }
      });

      // Populate equipment options first
      select.empty().append('<option value="">Select Equipment</option>');

      // Check if equipmentData is available and has items
      if (equipmentData && equipmentData.length > 0) {
        equipmentData.forEach(equipment => {
          const categoryName = equipment.category ? equipment.category.name : '';
          const sku = equipment.sku ? ` (${equipment.sku})` : '';
          const displayText = `${equipment.name}${categoryName ? ' - ' + categoryName : ''}${sku}`;

          const option = $('<option></option>')
            .attr('value', equipment.id)
            .attr('data-rate', equipment.daily_rate || 0)
            .attr('data-sku', equipment.sku || '')
            .text(displayText);

          select.append(option);
        });
      } else {
        console.warn('No equipment data available. Equipment count:', equipmentData ? equipmentData.length : 0);
        // If equipment data not loaded yet, wait and retry
        if (!equipmentDataLoaded) {
          setTimeout(() => {
            if (equipmentData && equipmentData.length > 0) {
              initializeEquipmentSelect(itemWrapper);
            }
          }, 500);
          return;
        }
      }

      // Initialize Select2 after populating options
      try {
        select.select2({
          placeholder: 'Select Equipment',
          dropdownParent: itemWrapper.closest('.card').length ? itemWrapper.closest('.card') : $('body'),
          allowClear: false,
          width: '100%'
        });
        console.log('Select2 initialized for equipment select');
      } catch (e) {
        console.error('Error initializing Select2:', e);
        // Retry after a short delay if initialization fails
        setTimeout(() => {
          try {
            select.select2({
              placeholder: 'Select Equipment',
              dropdownParent: itemWrapper.closest('.card').length ? itemWrapper.closest('.card') : $('body'),
              allowClear: false,
              width: '100%'
            });
            console.log('Select2 initialized on retry');
          } catch (e2) {
            console.error('Select2 initialization failed on retry:', e2);
          }
        }, 200);
      }

      // Remove existing change handlers to prevent duplicates
      select.off('change.equipment');

      // Handle equipment selection - auto-fill daily rate
      select.on('change.equipment', function() {
        const selectedOption = $(this).find('option:selected');
        const dailyRate = parseFloat(selectedOption.attr('data-rate')) || 0;
        const itemWrapper = $(this).closest('[data-repeater-item]');

        // Update daily rate field (using invoice-item-price which is the actual field class)
        itemWrapper.find('.invoice-item-price').val(dailyRate.toFixed(2));

        // Update stock availability
        const shopId = $('#basicShop').val();
        const equipmentId = $(this).val();
        if (shopId && equipmentId && typeof updateStockAvailability === 'function') {
          updateStockAvailability(itemWrapper, shopId, equipmentId);
        }

        // Calculate item total which will update subtotal display and grand total
        if (typeof calculateItemTotal === 'function') {
          calculateItemTotal(itemWrapper);
        } else if (typeof calculateTotal === 'function') {
          calculateTotal();
        }
      });
    }

    // Function to update stock availability display
    function updateStockAvailability(itemWrapper, shopId, equipmentId) {
      if (stockData[equipmentId] !== undefined) {
        itemWrapper.find('.available-stock').text(stockData[equipmentId]);
      } else {
        fetch(`/api/stocks?shop_id=${shopId}`)
          .then(response => response.json())
          .then(data => {
            if (data.success) {
              const stocks = Array.isArray(data.data) ? data.data : (data.data.data || []);
              stocks.forEach(stock => {
                stockData[stock.equipment_id] = stock.quantity - stock.reserved;
              });
              itemWrapper.find('.available-stock').text(stockData[equipmentId] || 0);
            }
          });
      }
    }

    // Function to format currency
    function formatCurrency(amount) {
      return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: 'LKR',
        minimumFractionDigits: 2
      }).format(amount);
    }

    // Function to calculate item total
    function calculateItemTotal(itemWrapper) {
      const qty = parseFloat(itemWrapper.find('.invoice-item-qty').val()) || 0;
      const days = parseFloat(itemWrapper.find('.invoice-item-days').val()) || 0;
      const rate = parseFloat(itemWrapper.find('.invoice-item-price').val()) || parseFloat(itemWrapper.find('.invoice-item-rate').val()) || 0;
      const subtotal = qty * days * rate;
      itemWrapper.find('.item-subtotal').text(formatCurrency(subtotal));
      calculateTotal();
    }

    // Function to calculate total
    function calculateTotal() {
      let subtotal = 0;
      $('#rentalForm').find('[data-repeater-item]').each(function() {
        const qty = parseFloat($(this).find('.invoice-item-qty').val()) || 0;
        const days = parseFloat($(this).find('.invoice-item-days').val()) || 0;
        const rate = parseFloat($(this).find('.invoice-item-price').val()) || parseFloat($(this).find('.invoice-item-rate').val()) || 0;
        subtotal += qty * days * rate;
      });

      const discount = parseFloat($('#discount').val()) || 0;
      const discountAmount = subtotal * (discount / 100);
      const grandTotal = subtotal - discountAmount;

      $('.subtotal-amount').text(formatCurrency(subtotal));
      $('.discount-amount').text(formatCurrency(discountAmount));
      $('.grand-total-amount').text(formatCurrency(grandTotal));
    }

    // Function to load equipment from stocks based on shop_id
    function loadEquipmentForShop(shopId) {
      if (!shopId) {
        equipmentData = [];
        equipmentDataLoaded = false;
        // Clear all equipment selects
        $('#rentalForm').find('[data-repeater-item]').each(function() {
          const select = $(this).find('.item-details');
          if (select.length) {
            try {
              select.select2('destroy');
            } catch (e) {
              console.warn('Error destroying Select2:', e);
            }
            select.empty().append('<option value="">Select Equipment</option>');
          }
        });
        return Promise.resolve();
      }

      console.log('Loading equipment for shop ID:', shopId);

      // Fetch all stocks for the shop
      return fetch(`/api/stocks?shop_id=${shopId}&all=true`)
        .then(response => {
          if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
          }
          return response.json();
        })
        .then(data => {
          console.log('Stocks API response:', data);
          if (data.success && data.data) {
            // Extract equipment from stocks
            const stocks = Array.isArray(data.data) ? data.data : [];

            console.log('Found stocks:', stocks.length);

            // Extract unique equipment from stocks
            const equipmentMap = new Map();
            stocks.forEach(stock => {
              if (stock.equipment && stock.equipment.id) {
                if (!equipmentMap.has(stock.equipment.id)) {
                  equipmentMap.set(stock.equipment.id, {
                    ...stock.equipment,
                    category: stock.equipment.category || null
                  });
                }
              }
            });
            equipmentData = Array.from(equipmentMap.values());

            // Store stock data for availability checks
            stockData = {};
            stocks.forEach(stock => {
              if (stock.equipment_id) {
                stockData[stock.equipment_id] = stock.quantity - stock.reserved;
              }
            });

            console.log('Loaded equipment for shop:', equipmentData.length, 'items');
            console.log('Equipment data:', equipmentData);
            equipmentDataLoaded = true;

            // Re-initialize all equipment selects with new data
            const repeaterItems = $('#rentalForm').find('[data-repeater-item]');
            console.log('Found repeater items:', repeaterItems.length);

            if (repeaterItems.length > 0) {
              repeaterItems.each(function() {
                initializeEquipmentSelect($(this));
              });
            } else {
              console.warn('No repeater items found to populate equipment');
            }
          } else {
            console.warn('No equipment data in response');
            equipmentData = [];
            equipmentDataLoaded = false;
          }
        })
        .catch(error => {
          console.error('Error fetching equipment for shop:', error);
          alert('Error loading equipment: ' + error.message);
          equipmentData = [];
          equipmentDataLoaded = false;
        });
    }

    document.addEventListener('DOMContentLoaded', function() {
      // Set rental ID display
      $('#rentalIdDisplay').val({{ $nextRentalId }});

      // Clean up any duplicate Select2 containers that might exist
      $('.select2-container').each(function() {
        const $container = $(this);
        const $prev = $container.prev();
        // Remove if there's no corresponding select element
        if ($prev.length === 0 || !$prev.is('select')) {
          $container.remove();
        }
      });

      // Prevent app-invoice-edit.js from initializing repeater on our form
      // Check and destroy any existing repeater instances
      setTimeout(() => {
        $('.source-item').not('#rentalForm').each(function() {
          if ($(this).data('repeater')) {
            try {
              $(this).repeater('destroy');
              $(this).removeData('repeater');
            } catch (e) {
              console.warn('Error cleaning up repeater:', e);
            }
          }
        });
      }, 100);

      // Helper function to safely initialize Select2
      function initializeSelect2(selector, options) {
        const $element = $(selector);
        if ($element.length === 0) return;

        // Destroy existing Select2 instance if it exists
        if ($element.hasClass('select2-hidden-accessible')) {
          try {
            $element.select2('destroy');
          } catch (e) {
            console.warn('Error destroying Select2:', e);
          }
          // Remove any leftover containers
          $element.next('.select2-container').remove();
          $element.removeClass('select2-hidden-accessible');
          $element.removeAttr('data-select2-id');
          $element.removeAttr('aria-hidden');
          $element.removeAttr('tabindex');
        }

        // Remove any duplicate containers
        $('.select2-container').filter(function() {
          return $(this).prev(selector).length > 0;
        }).remove();

        // Initialize Select2 with options
        $element.select2(options);
      }

      // Initialize Select2 for all dropdowns
      @php
        $userShopId = $user && !$user->isAdmin() ? $user->shop_id : null;
        $isAdmin = $user && $user->isAdmin();
      @endphp
      @if($userShopId)
        // Auto-select user's shop for non-admin users
        $('#basicShop').val('{{ $userShopId }}');
      @endif

      // Initialize shop dropdown
      initializeSelect2('#basicShop', {
        placeholder: 'Select Shop',
        dropdownParent: $('#basicShop').closest('.card').length ? $('#basicShop').closest('.card') : $('body'),
        allowClear: false
      });

      // Set up shop change handler BEFORE setting value or initializing repeater
      // Handle both regular change and Select2 change events
      function handleShopChange() {
        const shopId = $('#basicShop').val();
        console.log('Shop changed to:', shopId);
        if (shopId) {
          // Load equipment for the selected shop
          loadEquipmentForShop(shopId).then(() => {
            console.log('Equipment loaded, reinitializing selects');
            // If repeater is already initialized, update existing items
            if (repeaterInitialized) {
              $('#rentalForm').find('[data-repeater-item]').each(function() {
                initializeEquipmentSelect($(this));
              });
            } else {
              // Initialize repeater if not already done
              initializeRepeater();
            }
          });
        } else {
          // Clear equipment if no shop selected
          equipmentData = [];
          equipmentDataLoaded = false;
          stockData = {};
          // Clear all equipment selects
          $('#rentalForm').find('[data-repeater-item]').each(function() {
            const select = $(this).find('.item-details');
            if (select.length) {
              try {
                select.select2('destroy');
              } catch (e) {
                console.warn('Error destroying Select2:', e);
              }
              select.empty().append('<option value="">Select Equipment</option>');
            }
          });
        }
      }

      // Set up shop change handler - handle both Select2 and regular change events
      $('#basicShop').off('change.shopEquipment select2:select.shopEquipment').on('change.shopEquipment select2:select.shopEquipment', function(e) {
        e.stopPropagation();
        handleShopChange();
      });

      // Also listen for Select2 closing to ensure change event fires
      $('#basicShop').on('select2:close', function() {
        // Small delay to ensure value is set
        setTimeout(() => {
          const shopId = $(this).val();
          if (shopId && (!equipmentDataLoaded || equipmentData.length === 0)) {
            handleShopChange();
          }
        }, 100);
      });

      // Ensure shop dropdown closes after selection
      $('#basicShop').off('select2:select.close').on('select2:select.close', function() {
        $(this).trigger('select2:close');
      });

      // Initialize customer dropdown
      initializeSelect2('#basicCustomer', {
        placeholder: 'Select Customer (Optional)',
        dropdownParent: $('#basicCustomer').closest('.card').length ? $('#basicCustomer').closest('.card') : $('body'),
        allowClear: true
      });

      // Ensure customer dropdown closes after selection
      $('#basicCustomer').off('select2:select').on('select2:select', function() {
        $(this).trigger('select2:close');
      });

      // Load equipment when shop is selected (initial load for non-admin users)
      @if($userShopId)
        const initialShopId = $('#basicShop').val();
        if (initialShopId) {
          // Trigger change event to load equipment
          setTimeout(() => {
            handleShopChange();
          }, 100);
        } else {
          // Initialize repeater even if no shop selected
          if (!repeaterInitialized) {
            initializeRepeater();
          }
        }
      @else
        // For admin users, initialize repeater without equipment (will load when shop is selected)
        if (!repeaterInitialized) {
          initializeRepeater();
        }
      @endif

      // Initialize customer form validation and handling
      const rentalCustomerForm = document.getElementById('rentalCustomerForm');
      let rentalCustomerUploadedDocuments = [];

      if (rentalCustomerForm) {
        rentalCustomerForm.addEventListener('submit', function(e) {
          e.preventDefault();
        });

        // Profile image preview for rental customer form
        document.getElementById('rentalCustomerProfileImage')?.addEventListener('change', function(e) {
          const file = e.target.files[0];
          const preview = document.getElementById('rentalCustomerProfileImagePreview');
          if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
              preview.innerHTML = `<img src="${e.target.result}" class="img-thumbnail" style="max-width: 150px; max-height: 150px;" />`;
            };
            reader.readAsDataURL(file);
          } else {
            preview.innerHTML = '';
          }
        });

        // Document upload handling for rental customer form
        document.getElementById('rentalCustomerDocumentFiles')?.addEventListener('change', function(e) {
          const files = Array.from(e.target.files);
          const documentType = document.getElementById('rentalCustomerDocumentType').value;

          if (!documentType) {
            alert('Please select a document type first');
            e.target.value = '';
            return;
          }

          files.forEach(file => {
            if (file.size > 1024 * 1024) {
              alert(`File ${file.name} exceeds 1MB limit`);
              return;
            }

            const reader = new FileReader();
            reader.onload = function(e) {
              rentalCustomerUploadedDocuments.push({
                file: file,
                type: documentType,
                preview: e.target.result,
                name: file.name
              });
              window.displayRentalCustomerUploadedDocuments();
            };
            reader.readAsDataURL(file);
          });

          e.target.value = '';
        });

        // Display uploaded documents for rental customer form
        window.displayRentalCustomerUploadedDocuments = function() {
          const container = document.getElementById('rentalCustomerUploadedDocuments');
          if (!container) return;
          if (rentalCustomerUploadedDocuments.length === 0) {
            container.innerHTML = '';
            return;
          }

          let html = '<div class="list-group">';
          rentalCustomerUploadedDocuments.forEach((doc, index) => {
            const isImage = doc.file ? doc.file.type.startsWith('image/') : (doc.file_type === 'image' || doc.preview);
            const previewUrl = doc.preview || (doc.existing ? `/storage/${doc.file_path}` : null);
            const fileUrl = doc.existing ? `/api/documents/${doc.id}` : '#';

            html += `
              <div class="list-group-item d-flex justify-content-between align-items-center">
                <a href="${fileUrl}" target="_blank" class="d-flex align-items-center text-decoration-none flex-grow-1" style="color: inherit;">
                  ${isImage && previewUrl ? `<img src="${previewUrl}" style="width: 40px; height: 40px; object-fit: cover; margin-right: 10px;" class="rounded" />` : '<i class="icon-base ri ri-file-pdf-line icon-20px me-2"></i>'}
                  <div>
                    <div class="fw-medium">${doc.name}</div>
                    <small class="text-muted">${doc.type.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}</small>
                  </div>
                </a>
                <button type="button" class="btn btn-sm btn-outline-danger ms-2" onclick="removeRentalCustomerDocument(${index})">
                  <i class="icon-base ri ri-delete-bin-line"></i>
                </button>
              </div>
            `;
          });
          html += '</div>';
          container.innerHTML = html;
        };

        // Remove document from rental customer form
        window.removeRentalCustomerDocument = function(index) {
          rentalCustomerUploadedDocuments.splice(index, 1);
          displayRentalCustomerUploadedDocuments();
        };

        // Initialize FormValidation if available
        if (typeof FormValidation !== 'undefined') {
          const customerFv = FormValidation.formValidation(rentalCustomerForm, {
            fields: {
              basicCustomerName: {
                validators: {
                  notEmpty: {
                    message: 'Customer name is required'
                  }
                }
              },
              basicPhone: {
                validators: {
                  notEmpty: {
                    message: 'Phone number is required'
                  }
                }
              }
            },
            plugins: {
              trigger: new FormValidation.plugins.Trigger(),
              bootstrap5: new FormValidation.plugins.Bootstrap5({
                eleValidClass: '',
                rowSelector: '.form-control-validation'
              }),
              submitButton: new FormValidation.plugins.SubmitButton(),
              autoFocus: new FormValidation.plugins.AutoFocus()
            },
            init: instance => {
              instance.on('plugins.message.placed', function (e) {
                if (e.element.parentElement.classList.contains('input-group')) {
                  e.element.parentElement.insertAdjacentElement('afterend', e.messageElement);
                }
              });
            }
          });

          // Handle customer form submission
          customerFv.on('core.form.valid', function() {
            const formData = new FormData();
            formData.append('name', document.getElementById('rentalCustomerName').value);
            formData.append('phone', document.getElementById('rentalPhone').value);
            const nic = document.getElementById('rentalNic').value;
            const address = document.getElementById('rentalAddress').value;
            if (nic) formData.append('nic_number', nic);
            if (address) formData.append('address', address);

            // Add profile image if selected
            const profileImage = document.getElementById('rentalCustomerProfileImage')?.files[0];
            if (profileImage) {
              formData.append('profile_image', profileImage);
            }

            fetch('/api/customers', {
              method: 'POST',
              headers: {
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
              },
              credentials: 'include',
              body: formData
            })
              .then(response => {
                if (!response.ok) {
                  return response.json().then(err => {
                    throw new Error(err.message || JSON.stringify(err.errors || err));
                  });
                }
                return response.json();
              })
              .then(data => {
                if (data.success && data.data) {
                  const newCustomer = data.data;
                  const savedCustomerId = newCustomer.id;

                  // Upload documents if any
                  if (rentalCustomerUploadedDocuments && rentalCustomerUploadedDocuments.length > 0) {
                    const uploadPromises = rentalCustomerUploadedDocuments.map(doc => {
                      const docFormData = new FormData();
                      docFormData.append('model_type', 'App\\Models\\Customer');
                      docFormData.append('model_id', savedCustomerId);
                      docFormData.append('document_type', doc.type);
                      docFormData.append('document_name', doc.name);
                      docFormData.append('file', doc.file);

                      return fetch('/api/documents', {
                        method: 'POST',
                        headers: {
                          'Accept': 'application/json',
                          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        },
                        credentials: 'include',
                        body: docFormData
                      });
                    });

                    Promise.all(uploadPromises).then(() => {
                      // Add new customer to select dropdown
                      const newOption = new Option(
                        `${newCustomer.name} - ${newCustomer.phone}`,
                        newCustomer.id,
                        true,
                        true
                      );
                      $('#basicCustomer').append(newOption).trigger('change');

                      // Close offcanvas
                      const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('addCustomerOffcanvas'));
                      if (offcanvas) {
                        offcanvas.hide();
                      }

                      // Reset form
                      rentalCustomerForm.reset();
                      rentalCustomerUploadedDocuments = [];
                      window.displayRentalCustomerUploadedDocuments();
                      document.getElementById('rentalCustomerProfileImagePreview').innerHTML = '';

                      // Show success message
                      alert('Customer added successfully!');
                    }).catch(error => {
                      console.error('Error uploading documents:', error);
                      // Still add customer to dropdown even if documents fail
                      const newOption = new Option(
                        `${newCustomer.name} - ${newCustomer.phone}`,
                        newCustomer.id,
                        true,
                        true
                      );
                      $('#basicCustomer').append(newOption).trigger('change');

                      const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('addCustomerOffcanvas'));
                      if (offcanvas) {
                        offcanvas.hide();
                      }

                      rentalCustomerForm.reset();
                      rentalCustomerUploadedDocuments = [];
                      window.displayRentalCustomerUploadedDocuments();
                      document.getElementById('rentalCustomerProfileImagePreview').innerHTML = '';

                      alert('Customer added but some documents failed to upload');
                    });
                  } else {
                    // No documents to upload, just add customer to dropdown
                    const newOption = new Option(
                      `${newCustomer.name} - ${newCustomer.phone}`,
                      newCustomer.id,
                      true,
                      true
                    );
                    $('#basicCustomer').append(newOption).trigger('change');

                    // Close offcanvas
                    const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('addCustomerOffcanvas'));
                    if (offcanvas) {
                      offcanvas.hide();
                    }

                    // Reset form
                    rentalCustomerForm.reset();
                    rentalCustomerUploadedDocuments = [];
                    displayRentalCustomerUploadedDocuments();
                    document.getElementById('rentalCustomerProfileImagePreview').innerHTML = '';

                    // Show success message
                    alert('Customer added successfully!');
                  }
                } else {
                  alert('Error: ' + (data.message || 'Failed to add customer'));
                }
              })
              .catch(error => {
                console.error('Error:', error);
                alert('Error: ' + error.message);
              });

            return false;
          });
        } else {
          // Fallback if FormValidation is not available
          rentalCustomerForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const name = document.getElementById('rentalCustomerName').value;
            const phone = document.getElementById('rentalPhone').value;

            if (!name || !phone) {
              alert('Customer name and phone are required');
              return;
            }

            const formData = new FormData();
            formData.append('name', name);
            formData.append('phone', phone);
            const nic = document.getElementById('rentalNic').value;
            const address = document.getElementById('rentalAddress').value;
            if (nic) formData.append('nic_number', nic);
            if (address) formData.append('address', address);

            // Add profile image if selected
            const profileImage = document.getElementById('rentalCustomerProfileImage')?.files[0];
            if (profileImage) {
              formData.append('profile_image', profileImage);
            }

            fetch('/api/customers', {
              method: 'POST',
              headers: {
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
              },
              credentials: 'include',
              body: formData
            })
              .then(response => {
                if (!response.ok) {
                  return response.json().then(err => {
                    throw new Error(err.message || JSON.stringify(err.errors || err));
                  });
                }
                return response.json();
              })
              .then(data => {
                if (data.success && data.data) {
                  const newCustomer = data.data;
                  const savedCustomerId = newCustomer.id;

                  // Upload documents if any
                  if (rentalCustomerUploadedDocuments && rentalCustomerUploadedDocuments.length > 0) {
                    const uploadPromises = rentalCustomerUploadedDocuments.map(doc => {
                      const docFormData = new FormData();
                      docFormData.append('model_type', 'App\\Models\\Customer');
                      docFormData.append('model_id', savedCustomerId);
                      docFormData.append('document_type', doc.type);
                      docFormData.append('document_name', doc.name);
                      docFormData.append('file', doc.file);

                      return fetch('/api/documents', {
                        method: 'POST',
                        headers: {
                          'Accept': 'application/json',
                          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                        },
                        credentials: 'include',
                        body: docFormData
                      });
                    });

                    Promise.all(uploadPromises).then(() => {
                      // Add new customer to select dropdown
                      const newOption = new Option(
                        `${newCustomer.name} - ${newCustomer.phone}`,
                        newCustomer.id,
                        true,
                        true
                      );
                      $('#basicCustomer').append(newOption).trigger('change');

                      // Close offcanvas
                      const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('addCustomerOffcanvas'));
                      if (offcanvas) {
                        offcanvas.hide();
                      }

                      // Reset form
                      rentalCustomerForm.reset();
                      rentalCustomerUploadedDocuments = [];
                      window.displayRentalCustomerUploadedDocuments();
                      document.getElementById('rentalCustomerProfileImagePreview').innerHTML = '';

                      // Show success message
                      alert('Customer added successfully!');
                    }).catch(error => {
                      console.error('Error uploading documents:', error);
                      // Still add customer to dropdown even if documents fail
                      const newOption = new Option(
                        `${newCustomer.name} - ${newCustomer.phone}`,
                        newCustomer.id,
                        true,
                        true
                      );
                      $('#basicCustomer').append(newOption).trigger('change');

                      const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('addCustomerOffcanvas'));
                      if (offcanvas) {
                        offcanvas.hide();
                      }

                      rentalCustomerForm.reset();
                      rentalCustomerUploadedDocuments = [];
                      window.displayRentalCustomerUploadedDocuments();
                      document.getElementById('rentalCustomerProfileImagePreview').innerHTML = '';

                      alert('Customer added but some documents failed to upload');
                    });
                  } else {
                    // No documents to upload, just add customer to dropdown
                    const newOption = new Option(
                      `${newCustomer.name} - ${newCustomer.phone}`,
                      newCustomer.id,
                      true,
                      true
                    );
                    $('#basicCustomer').append(newOption).trigger('change');

                    // Close offcanvas
                    const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('addCustomerOffcanvas'));
                    if (offcanvas) {
                      offcanvas.hide();
                    }

                    // Reset form
                    rentalCustomerForm.reset();
                    rentalCustomerUploadedDocuments = [];
                    displayRentalCustomerUploadedDocuments();
                    document.getElementById('rentalCustomerProfileImagePreview').innerHTML = '';

                    // Show success message
                    alert('Customer added successfully!');
                  }
                } else {
                  alert('Error: ' + (data.message || 'Failed to add customer'));
                }
              })
              .catch(error => {
                console.error('Error:', error);
                alert('Error: ' + error.message);
              });
          });
        }
      }

      // Handle add customer button click
      document.getElementById('addCustomerBtn')?.addEventListener('click', function() {
        const offcanvas = new bootstrap.Offcanvas(document.getElementById('addCustomerOffcanvas'));
        if (rentalCustomerForm) {
          rentalCustomerForm.reset();
          rentalCustomerUploadedDocuments = [];
          displayRentalCustomerUploadedDocuments();
          document.getElementById('rentalCustomerProfileImagePreview').innerHTML = '';
        }
        offcanvas.show();
      });

      $('#basicStatus').select2({
        placeholder: 'Select Status',
        dropdownParent: $('.card'),
        minimumResultsForSearch: Infinity
      });

      function initializeRepeater() {
        if (repeaterInitialized) {
          console.warn('Repeater already initialized');
          return;
        }

        // Destroy ALL existing repeater instances to prevent conflicts
        try {
          // Destroy any repeater that might have been initialized by app-invoice-edit.js
          $('.source-item').each(function() {
            if ($(this).data('repeater')) {
              $(this).repeater('destroy');
            }
          });

          // Also try to destroy by removing data attribute
          $('.source-item').removeData('repeater');

          // Remove any existing event handlers on the create button
          $('[data-repeater-create]').off();
        } catch (e) {
          console.warn('Error destroying existing repeater:', e);
        }

        // Wait a bit to ensure cleanup is complete
        setTimeout(() => {
          // Double-check that no repeater exists
          const rentalForm = $('#rentalForm');
          if (rentalForm.data('repeater')) {
            console.warn('Repeater still exists, forcing destroy');
            try {
              rentalForm.repeater('destroy');
              rentalForm.removeData('repeater');
            } catch (e) {
              console.error('Force destroy failed:', e);
            }
          }

          // Initialize repeater for equipment items - use specific selector to avoid conflicts
          rentalForm.repeater({
            show: function() {
              const itemWrapper = $(this);
              $(this).slideDown(function() {
                // Wait for slideDown animation to complete, then initialize
                setTimeout(() => {
                  // Clean up any existing Select2 instances first
                  itemWrapper.find('.item-details').each(function() {
                    const $select = $(this);
                    if ($select.hasClass('select2-hidden-accessible') || $select.next(
                        '.select2-container').length > 0) {
                      try {
                        $select.select2('destroy');
                      } catch (e) {
                        // Ignore errors
                      }
                      $select.next('.select2-container').remove();
                      $select.removeClass('select2-hidden-accessible');
                      $select.removeAttr('data-select2-id');
                    }
                  });
                  // Now initialize Select2 for the new row
                  initializeEquipmentSelect(itemWrapper);
                }, 300);
              });
            },
            hide: function(deleteElement) {
              if (confirm('Are you sure you want to delete this item?')) {
                $(this).slideUp(deleteElement);
                calculateTotal();
              }
            },
            ready: function(setIndexes) {
              // Initialize first item after equipment data is loaded
              if (equipmentDataLoaded) {
                setTimeout(() => {
                  const firstItem = $('#rentalForm').find('.repeater-wrapper').first();
                  if (firstItem.length > 0) {
                    initializeEquipmentSelect(firstItem);
                  }
                }, 300);
              } else {
                // Wait for equipment data to load
                const checkInterval = setInterval(() => {
                  if (equipmentDataLoaded) {
                    clearInterval(checkInterval);
                    const firstItem = $('#rentalForm').find('.repeater-wrapper').first();
                    if (firstItem.length > 0) {
                      initializeEquipmentSelect(firstItem);
                    }
                  }
                }, 100);

                // Timeout after 5 seconds to prevent infinite waiting
                setTimeout(() => {
                  clearInterval(checkInterval);
                }, 5000);
              }
            }
          });

          repeaterInitialized = true;
          console.log('Repeater initialized successfully on #rentalForm');
        }, 200);
      }

      // Handle discount change
      $('#discount').on('input', calculateTotal);

      // Handle guarantee type change to show/hide guarantee details
      const guaranteeTypeSelect = document.getElementById('basicGuaranteeType');
      const guaranteeDetailsWrapper = document.getElementById('guaranteeDetailsWrapper');

      if (guaranteeTypeSelect && guaranteeDetailsWrapper) {
        guaranteeTypeSelect.addEventListener('change', function() {
          if (this.value === 'other') {
            guaranteeDetailsWrapper.style.display = 'block';
            document.getElementById('basicGuaranteeDetails').required = true;
          } else {
            guaranteeDetailsWrapper.style.display = 'none';
            document.getElementById('basicGuaranteeDetails').required = false;
            document.getElementById('basicGuaranteeDetails').value = '';
          }
        });
      }

      // Handle form submission
      $('#rentalForm').on('submit', function(e) {
        e.preventDefault();

        const formData = {
          shop_id: $('#basicShop').val(),
          customer_id: $('#basicCustomer').val() || null,
          rented_by: {{ auth()->id() ?? 1 }},
          status: $('#basicStatus').val(),
          rental_date: $('#basicRentalDate').val(),
          expected_return_date: $('#basicExpectedReturn').val() || null,
          remarks: $('#basicRemarks').val() || null,
          guarantee_type: $('#basicGuaranteeType').val() || null,
          guarantee_details: $('#basicGuaranteeType').val() === 'other' ? ($('#basicGuaranteeDetails').val() || null) : null,
          discount: parseFloat($('#discount').val()) || 0,
          items: []
        };

        // Collect items
        $('#rentalForm').find('[data-repeater-item]').each(function() {
          const equipmentId = $(this).find('.item-details').val();
          if (equipmentId) {
            formData.items.push({
              equipment_id: equipmentId,
              quantity: parseInt($(this).find('.invoice-item-qty').val()) || 1,
              days: parseFloat($(this).find('.invoice-item-days').val()) || 1,
              serial_numbers: null
            });
          }
        });

        if (formData.items.length === 0) {
          alert('Please add at least one equipment item');
          return;
        }

        const url = currentRentalId ? `/api/rentals/${currentRentalId}` : '/api/rentals';
        const method = currentRentalId ? 'PUT' : 'POST';

        fetch(url, {
            method: method,
            headers: {
              'Content-Type': 'application/json',
              'Accept': 'application/json',
              'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            body: JSON.stringify(formData)
          })
          .then(response => response.json())
          .then(data => {
            if (data.success) {
              if (data.data.agreement_id) {
                // Redirect to agreement preview page where user can print
                window.location.href = `/agreements/${data.data.agreement_id}/preview`;
              } else {
                // If no agreement created, redirect to rentals list
                window.location.href = '{{ route('rentals') }}';
              }
            } else {
              alert('Error: ' + (data.message || 'Failed to save rental'));
            }
          })
          .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while saving the rental');
          });
      });

      // Initialize status dropdown (basicShop and basicCustomer are already initialized above)
      initializeSelect2('#basicStatus', {
        placeholder: 'Select Status',
        dropdownParent: $('#basicStatus').closest('.card').length ? $('#basicStatus').closest('.card') : $('body'),
        allowClear: true
      });

      // Ensure status dropdown closes after selection
      $('#basicStatus').off('select2:select').on('select2:select', function() {
        $(this).trigger('select2:close');
      });

      // Set default rental date to current date/time
      const now = new Date();
      now.setMinutes(now.getMinutes() - now.getTimezoneOffset());
      $('#basicRentalDate').val(now
        .toISOString().slice(0, 16));
    });
  </script>
@endsection

@section('content')
  <div class="row invoice-edit">
    <!-- Rental Edit-->
    <div class="col-lg-9 col-12 mb-lg-0 mb-6">
      <div class="card invoice-preview-card p-sm-12 p-6">
        <div class="card-body invoice-preview-header rounded p-6 px-3 text-heading">
          <div class="row mx-0 px-3">
            <div class="col-md-7 mb-md-0 mb-6 ps-0">
              <div class="d-flex svg-illustration align-items-center gap-3 mb-3">
                <a href="{{ url('/') }}" class="app-brand-link gap-2">
                  <img src="{{ asset('assets/img/branding/logo.png') }}" alt="brand-logo" width="200"
                    class="rounded-4" />
                </a>
              </div>
              <h4 class="mb-1">Equipment Renting System</h4>
            </div>
            <div class="col-md-5 col-8 pe-0 ps-0 ps-md-2">
              <dl class="row mb-0 gx-4">
                <dt class="col-sm-5 mb-2 d-md-flex align-items-center justify-content-start">
                  <span class="h5 text-capitalize mb-0 text-nowrap">Rental</span>
                </dt>
                <dd class="col-sm-7">
                  <div class="input-group input-group-sm input-group-merge disabled">
                    <span class="input-group-text">#</span>
                    <input type="text" class="form-control" disabled placeholder="{{ $nextRentalId }}"
                      value="{{ $nextRentalId }}" id="rentalIdDisplay" />
                  </div>
                </dd>
                <dt class="col-sm-5 mb-2 d-md-flex align-items-center justify-content-start">
                  <span class="fw-normal text-nowrap">Rental Date:</span>
                </dt>
                <dd class="col-sm-7">
                  <input type="datetime-local" class="form-control form-control-sm" id="basicRentalDate" />
                </dd>
                <dt class="col-sm-5 d-md-flex align-items-center justify-content-start">
                  <span class="fw-normal">Expected Return:</span>
                </dt>
                <dd class="col-sm-7 mb-0">
                  <input type="datetime-local" class="form-control form-control-sm" id="basicExpectedReturn" />
                </dd>
              </dl>
            </div>
          </div>
        </div>

        <div class="card-body px-0">
          <div class="row my-1">
            <div class="col-md-6 col-sm-5 col-12 mb-sm-0 mb-6">
              <h6>Shop:</h6>
              <select class="select2 form-select mb-4 w-auto" id="basicShop" @if(!$user || !$user->isAdmin()) disabled @endif>
                <option value="">Select Shop</option>
                @foreach ($shops as $shop)
                  <option value="{{ $shop->id }}" @if($user && !$user->isAdmin() && $user->shop_id == $shop->id) selected @endif>{{ $shop->name }}</option>
                @endforeach
              </select>
            </div>
            <div class="col-md-6 col-sm-7">
              <h6>Customer:
                <button type="button" class="btn btn-sm btn-icon btn-outline-primary ms-2" id="addCustomerBtn" title="Add New Customer">
                  <i class="icon-base ri ri-add-line"></i>
                </button>
              </h6>
              <select class="form-select mb-4 w-auto" id="basicCustomer">
                <option value="">Select Customer (Optional)</option>
                @foreach ($customers as $customer)
                  <option value="{{ $customer->id }}">{{ $customer->name }} - {{ $customer->phone }}</option>
                @endforeach
              </select>
              <div class="mt-2">
                <label class="form-label">Status:</label>
                <select class="select2 form-select" id="basicStatus">
                  <option value="draft">Draft</option>
                  <option value="active">Active</option>
                </select>
              </div>
            </div>
          </div>
        </div>
        <hr class="mb-6 mt-1" />
        <div class="card-body pt-0 px-0">
          <form class="source-item" id="rentalForm">
            <div class="mb-4" data-repeater-list="group-a">
              <div class="repeater-wrapper pt-0 pt-md-9" data-repeater-item>
                <div class="d-flex border rounded position-relative pe-0">
                  <div class="row w-100 p-5 gx-5">
                    <div class="col-md-5 col-12 mb-md-0 mb-3">
                      <h6 class="mb-2 repeater-title fw-medium">Equipment</h6>
                      <select class="form-select item-details mb-4" data-select2-init="false">
                        <option value="">Select Equipment</option>
                      </select>
                      <textarea class="form-control" rows="2" placeholder="Notes (optional)"></textarea>
                    </div>
                    <div class="col-md-2 col-12 mb-md-0 mb-5">
                      <h6 class="h6 repeater-title">Rate/Day</h6>
                      <input type="text" class="form-control invoice-item-price mb-5" value="0" placeholder="0"
                        readonly />
                      <div class="d-flex flex-column gap-2 text-heading">
                        <span>Available: <span class="available-stock">-</span></span>
                      </div>
                    </div>
                    <div class="col-md-2 col-12 mb-md-0 mb-4">
                      <h6 class="h6 repeater-title">Qty</h6>
                      <input type="number" class="form-control invoice-item-qty" value="1" placeholder="1"
                        min="1" />
                    </div>
                    <div class="col-md-2 col-12 mb-md-0 mb-4">
                      <h6 class="h6 repeater-title">Days</h6>
                      <input type="number" class="form-control invoice-item-days" value="1" placeholder="1"
                        min="0.5" step="0.5" />
                    </div>
                    <div class="col-md-1 col-12 pe-0">
                      <h6 class="h6 repeater-title">Subtotal</h6>
                      <p class="mb-0 mt-2 text-heading item-subtotal">LKR 0.00</p>
                    </div>
                  </div>
                  <div class="d-flex flex-column align-items-center justify-content-between border-start p-2">
                    <i class="icon-base ri ri-close-line cursor-pointer icon-24px" data-repeater-delete></i>
                  </div>
                </div>
              </div>
            </div>
            <div class="row">
              <div class="col-12">
                <button type="button" class="btn btn-primary btn-sm" data-repeater-create><i
                    class="icon-base ri ri-add-line icon-14px me-1"></i> Add Equipment</button>
              </div>
            </div>
          </form>
        </div>
        <hr class="my-1" />
        <div class="card-body px-0">
          <div class="row">
            <div class="col-md-6 mb-md-0 mb-3">
              <div class="d-flex align-items-center mb-4">
                <label for="remarks" class="me-2 h6 mb-2">Remarks:</label>
              </div>
              <textarea class="form-control" rows="3" id="basicRemarks" placeholder="Additional notes..."></textarea>
              <div class="mt-4">
                <label class="form-label h6 mb-2">Guarantee Type:</label>
                <select class="form-select mb-3" id="basicGuaranteeType">
                  <option value="">Select Guarantee (Optional)</option>
                  <option value="cash">Cash</option>
                  <option value="nic">NIC</option>
                  <option value="driving_license">Driving License</option>
                  <option value="other">Other</option>
                </select>
                <div id="guaranteeDetailsWrapper" style="display: none;">
                  <label class="form-label">Guarantee Details:</label>
                  <textarea class="form-control" rows="3" id="basicGuaranteeDetails" placeholder="Specify guarantee details..."></textarea>
                </div>
              </div>
            </div>
            <div class="col-md-6 d-flex justify-content-md-end mt-2">
              <div class="invoice-calculations">
                <div class="d-flex justify-content-between mb-1">
                  <span class="w-px-100">Subtotal:</span>
                  <h6 class="mb-0 subtotal-amount">LKR 0.00</h6>
                </div>
                <div class="d-flex justify-content-between mb-1">
                  <span class="w-px-100">Discount (%):</span>
                  <input type="number" class="form-control form-control-sm w-px-100" id="discount" value="0"
                    min="0" max="100" />
                </div>
                <div class="d-flex justify-content-between mb-1">
                  <span class="w-px-100">Discount:</span>
                  <h6 class="mb-0 discount-amount">LKR 0.00</h6>
                </div>
                <hr class="my-2" />
                <div class="d-flex justify-content-between">
                  <span class="w-px-100">Total:</span>
                  <h6 class="mb-0 grand-total-amount">LKR 0.00</h6>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
    <!-- /Rental Edit-->

    <!-- Rental Actions -->
    <div class="col-lg-3 col-12 invoice-actions">
      <div class="card mb-6">
        <div class="card-body">
          <button type="submit" form="rentalForm" class="btn btn-primary d-grid w-100 mb-4">
            <span class="d-flex align-items-center justify-content-center text-nowrap">
              <i class="icon-base ri ri-save-line icon-16px scaleX-n1-rtl me-2"></i>Save Rental</span>
          </button>
          <div class="d-flex">
            <a href="{{ route('rentals') }}" class="btn btn-outline-secondary w-100 me-4 mb-4">Cancel</a>
          </div>
          <button type="button" class="btn btn-success d-grid w-100" id="generateInvoiceBtn" disabled>
            <span class="d-flex align-items-center justify-content-center text-nowrap">
              <i class="icon-base ri ri-file-pdf-line icon-16px scaleX-n1-rtl me-2"></i>Generate Invoice</span>
          </button>
        </div>
      </div>
    </div>
    <!-- /Rental Actions -->
  </div>

  <!-- Add Customer Offcanvas -->
  <div class="offcanvas offcanvas-end" tabindex="-1" id="addCustomerOffcanvas" aria-labelledby="addCustomerOffcanvasLabel">
    <div class="offcanvas-header border-bottom">
      <h5 class="offcanvas-title" id="addCustomerOffcanvasLabel">Add New Customer</h5>
      <button type="button" class="btn-close text-reset" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body flex-grow-1">
      <form method="POST" class="add-new-record pt-0 row g-3" id="rentalCustomerForm">
        @csrf
        <input type="hidden" id="rentalCustomerId" name="customerId" />

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="rentalCustomerName2" class="input-group-text">
              <i class="icon-base ri ri-user-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <input type="text" id="rentalCustomerName" class="form-control" name="basicCustomerName"
                     placeholder="Customer Name" aria-label="Customer Name" aria-describedby="rentalCustomerName2" />
              <label for="rentalCustomerName">Customer Name</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="rentalPhone2" class="input-group-text">
              <i class="icon-base ri ri-phone-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <input type="tel" id="rentalPhone" name="basicPhone" class="form-control"
                     placeholder="Phone Number" aria-label="Phone Number" aria-describedby="rentalPhone2" />
              <label for="rentalPhone">Phone Number</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="rentalNic2" class="input-group-text">
              <i class="icon-base ri ri-id-card-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <input type="text" id="rentalNic" name="basicNic" class="form-control"
                     placeholder="NIC Number" aria-label="NIC Number" aria-describedby="rentalNic2" />
              <label for="rentalNic">NIC Number (Optional)</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12 form-control-validation">
          <div class="input-group input-group-merge">
            <span id="rentalAddress2" class="input-group-text">
              <i class="icon-base ri ri-map-pin-line icon-18px"></i>
            </span>
            <div class="form-floating form-floating-outline">
              <textarea class="form-control" id="rentalAddress" name="basicAddress"
                        placeholder="Address" aria-label="Address" aria-describedby="rentalAddress2" rows="3"></textarea>
              <label for="rentalAddress">Address (Optional)</label>
            </div>
          </div>
        </div>

        <div class="col-sm-12">
          <label class="form-label">Profile Image</label>
          <input type="file" id="rentalCustomerProfileImage" name="profile_image" class="form-control" accept="image/jpeg,image/jpg,image/png" />
          <small class="text-muted">Max size: 1MB (JPG, PNG only)</small>
          <div id="rentalCustomerProfileImagePreview" class="mt-2"></div>
        </div>

        <div class="col-sm-12 mt-4">
          <label class="form-label">Documents</label>
          <div class="mb-3">
            <select id="rentalCustomerDocumentType" class="form-select form-select-sm mb-2">
              <option value="">Select Document Type</option>
              <option value="nic">NIC</option>
              <option value="passport">Passport</option>
              <option value="driving_license">Driving License</option>
              <option value="other">Other</option>
            </select>
            <input type="file" id="rentalCustomerDocumentFiles" class="form-control" accept=".jpg,.jpeg,.png,.pdf" multiple />
            <small class="text-muted">Max size: 1MB per file (JPG, PNG, PDF only)</small>
          </div>
          <div id="rentalCustomerUploadedDocuments" class="mt-3"></div>
        </div>

        <div class="col-sm-12">
          <button type="submit" id="rentalCustomerSubmitBtn" class="btn btn-primary data-submit me-sm-4 me-1">Add Customer</button>
          <button type="reset" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">Cancel</button>
        </div>
      </form>
    </div>
  </div>
  <!-- /Add Customer Offcanvas -->
@endsection
