@extends('layouts/layoutMaster')

@section('title', 'Permissions Management - Equipment Renting System')

<!-- Vendor Styles -->
@section('vendor-style')
  @vite(['resources/assets/vendor/libs/select2/select2.scss'])
@endsection

<!-- Vendor Scripts -->
@section('vendor-script')
  @vite(['resources/assets/vendor/libs/select2/select2.js'])
@endsection

@section('page-style')
  <style>
    .permission-group {
      margin-bottom: 2rem;
    }
    .permission-group-title {
      font-size: 1.125rem;
      font-weight: 600;
      margin-bottom: 1rem;
      padding-bottom: 0.5rem;
      border-bottom: 2px solid #e4e6ef;
      text-transform: capitalize;
    }
    .permission-group table {
      margin-bottom: 0;
    }
    .permission-group table th {
      background-color: #f5f5f9;
      font-weight: 600;
      font-size: 0.875rem;
      text-transform: uppercase;
      color: #697a8d;
    }
    .permission-group table td {
      vertical-align: middle;
    }
    .switch {
      position: relative;
      display: inline-block;
      width: 48px;
      height: 24px;
    }
    .switch-input {
      opacity: 0;
      width: 0;
      height: 0;
    }
    .switch-toggle-slider {
      position: absolute;
      cursor: pointer;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background-color: #ccc;
      transition: .4s;
      border-radius: 24px;
    }
    .switch-toggle-slider:before {
      position: absolute;
      content: "";
      height: 18px;
      width: 18px;
      left: 3px;
      bottom: 3px;
      background-color: white;
      transition: .4s;
      border-radius: 50%;
    }
    .switch-input:checked + .switch-toggle-slider {
      background-color: #696cff;
    }
    .switch-input:checked + .switch-toggle-slider:before {
      transform: translateX(24px);
    }
    .switch-input:disabled + .switch-toggle-slider {
      opacity: 0.6;
      cursor: not-allowed;
    }
    #permissionsContainer {
      display: none;
    }
    .no-user-selected {
      text-align: center;
      padding: 3rem;
      color: #a1acb8;
    }
  </style>
@endsection

@section('content')
  <meta name="csrf-token" content="{{ csrf_token() }}">
  
  <div class="container-xxl flex-grow-1 container-p-y">
    <h4 class="fw-bold py-3 mb-4">
      <span class="text-muted fw-light">Settings /</span> Permissions Management
    </h4>

    <div class="card">
      <div class="card-header">
        <h5 class="card-title mb-0">Manage User Permissions</h5>
      </div>
      <div class="card-body">
        <!-- User Selection -->
        <div class="row mb-4">
          <div class="col-md-6">
            <label class="form-label">Select User</label>
            <select class="select2 form-select" id="userSelect">
              <option value="">-- Select User --</option>
              @foreach ($users as $user)
                <option value="{{ $user->id }}" data-is-admin="{{ $user->isAdmin() ? '1' : '0' }}">
                  {{ $user->name }} ({{ $user->email }})
                  @if($user->isAdmin())
                    - Admin
                  @endif
                </option>
              @endforeach
            </select>
          </div>
        </div>

        <!-- Permissions Container -->
        <div id="permissionsContainer">
          <div id="selectedUserInfo" class="alert alert-info mb-4" style="display: none;">
            <strong>Selected User:</strong> <span id="selectedUserName"></span>
            <span id="adminWarning" class="text-warning ms-2" style="display: none;">
              (Admin users have all permissions by default)
            </span>
          </div>

          <div id="permissionsList">
            @foreach ($groupedPermissions as $module => $modulePermissions)
              <div class="permission-group">
                <div class="permission-group-title">
                  {{ ucfirst($module) }}
                </div>
                <div class="table-responsive">
                  <table class="table table-bordered">
                    <thead>
                      <tr>
                        <th style="width: 40%;">Permission / Description</th>
                        <th style="width: 12%; text-align: center;">Action</th>
                        <th style="width: 8%; text-align: center;">Status</th>
                      </tr>
                    </thead>
                    <tbody>
                      @foreach ($modulePermissions as $permission)
                        <tr>
                          <td>
                            <div>
                              <strong>{{ ucfirst(str_replace('_', ' ', $permission->action)) }}</strong>
                              @if($permission->description)
                                <br><small class="text-muted">{{ $permission->description }}</small>
                              @endif
                              <br><small class="text-info permission-source" id="source_{{ $permission->id }}" style="display: none;"></small>
                            </div>
                          </td>
                          <td style="text-align: center;">
                            @php
                              $badgeColor = 'secondary';
                              if (in_array($permission->action, ['view', 'total_rentals', 'active_rentals', 'total_revenue', 'today_revenue', 'total_customers', 'total_equipment', 'active_stock', 'revenue_chart', 'status_chart', 'equipment_chart', 'daily_rentals_chart'])) {
                                $badgeColor = 'primary';
                              } elseif (in_array($permission->action, ['create'])) {
                                $badgeColor = 'success';
                              } elseif (in_array($permission->action, ['update'])) {
                                $badgeColor = 'warning';
                              } elseif (in_array($permission->action, ['delete'])) {
                                $badgeColor = 'danger';
                              }
                            @endphp
                            <span class="badge bg-label-{{ $badgeColor }}">
                              {{ ucfirst(str_replace('_', ' ', $permission->action)) }}
                            </span>
                          </td>
                          <td style="text-align: center;">
                            <label class="switch">
                              <input 
                                type="checkbox" 
                                class="switch-input permission-toggle" 
                                data-permission-id="{{ $permission->id }}"
                                data-permission-name="{{ $permission->name }}"
                                id="permission_{{ $permission->id }}"
                              />
                              <span class="switch-toggle-slider"></span>
                            </label>
                          </td>
                        </tr>
                      @endforeach
                    </tbody>
                  </table>
                </div>
              </div>
            @endforeach
          </div>

          <div class="mt-4">
            <button type="button" class="btn btn-primary" id="savePermissionsBtn">
              <i class="icon-base ri ri-save-line me-2"></i>Save All Permissions
            </button>
          </div>
        </div>

        <div id="noUserSelected" class="no-user-selected">
          <i class="icon-base ri ri-user-line icon-48px mb-3"></i>
          <p>Please select a user to manage permissions</p>
        </div>
      </div>
    </div>
  </div>

  <script>
    'use strict';
    let currentUserId = null;
    let isAdmin = false;

    document.addEventListener('DOMContentLoaded', function() {
      // Initialize Select2
      $('#userSelect').select2({
        placeholder: 'Select User',
        allowClear: true,
        dropdownParent: $('.card')
      });

      // Handle user selection change
      $('#userSelect').on('change', function() {
        const userId = $(this).val();
        if (userId) {
          loadUserPermissions(userId);
        } else {
          resetPermissionsView();
        }
      });

      // Handle individual permission toggle
      $(document).on('change', '.permission-toggle', function() {
        if (currentUserId && !isAdmin) {
          const permissionId = $(this).data('permission-id');
          const granted = $(this).is(':checked');
          
          togglePermission(currentUserId, permissionId, granted);
        }
      });

      // Handle save all permissions button
      $('#savePermissionsBtn').on('click', function() {
        if (currentUserId && !isAdmin) {
          saveAllPermissions(currentUserId);
        }
      });
    });

    function loadUserPermissions(userId) {
      currentUserId = userId;
      const selectedOption = $('#userSelect option:selected');
      isAdmin = selectedOption.data('is-admin') == 1;

      // Show selected user info
      $('#selectedUserName').text(selectedOption.text().split(' -')[0]);
      $('#adminWarning').toggle(isAdmin);
      $('#selectedUserInfo').show();

      // Show permissions container
      $('#permissionsContainer').show();
      $('#noUserSelected').hide();

      // Disable all toggles if admin
      $('.permission-toggle').prop('disabled', isAdmin);

      if (isAdmin) {
        // Admin has all permissions - check all
        $('.permission-toggle').prop('checked', true);
        return;
      }

      // Fetch user permissions from API
      fetch(`/permissions/user/${userId}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          // Reset all checkboxes
          $('.permission-toggle').prop('checked', false);
          $('.permission-source').hide();
          
          // Set granted permissions and show source
          data.permissions.forEach(permission => {
            if (permission.granted) {
              $(`#permission_${permission.id}`).prop('checked', true);
            }
            
            // Show permission source (role or user-specific)
            const sourceElement = $(`#source_${permission.id}`);
            if (permission.from_role) {
              sourceElement.text(`(From ${data.user.role || 'Role'})`).show();
            } else {
              sourceElement.text('(User-specific override)').show();
            }
          });
        }
      })
      .catch(error => {
        console.error('Error loading permissions:', error);
        alert('Error loading user permissions');
      });
    }

    function togglePermission(userId, permissionId, granted) {
      fetch(`/permissions/user/${userId}/toggle`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({
          permission_id: permissionId,
          granted: granted
        })
      })
      .then(response => response.json())
      .then(data => {
        if (!data.success) {
          alert(data.message || 'Error updating permission');
          // Revert toggle
          $(`#permission_${permissionId}`).prop('checked', !granted);
        }
      })
      .catch(error => {
        console.error('Error toggling permission:', error);
        alert('Error updating permission');
        // Revert toggle
        $(`#permission_${permissionId}`).prop('checked', !granted);
      });
    }

    function saveAllPermissions(userId) {
      const permissions = {};
      $('.permission-toggle').each(function() {
        const permissionId = $(this).data('permission-id');
        permissions[permissionId] = $(this).is(':checked');
      });

      fetch(`/permissions/user/${userId}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({
          permissions: permissions
        })
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          alert('Permissions saved successfully!');
        } else {
          alert(data.message || 'Error saving permissions');
        }
      })
      .catch(error => {
        console.error('Error saving permissions:', error);
        alert('Error saving permissions');
      });
    }

    function resetPermissionsView() {
      currentUserId = null;
      isAdmin = false;
      $('#permissionsContainer').hide();
      $('#noUserSelected').show();
      $('#selectedUserInfo').hide();
      $('.permission-toggle').prop('checked', false).prop('disabled', false);
    }
  </script>
@endsection
