@extends('layouts/layoutMaster')

@section('title', 'Invoice - Equipment Renting System')

@section('vendor-style')
@vite('resources/assets/vendor/libs/flatpickr/flatpickr.scss')
@endsection

@section('page-style')
@vite('resources/assets/vendor/scss/pages/app-invoice.scss')
<style>
  @media print {
    .no-print {
      display: none !important;
    }
    .card {
      border: none !important;
      box-shadow: none !important;
    }
  }
</style>
@endsection

@section('vendor-script')
@vite([
'resources/assets/vendor/libs/flatpickr/flatpickr.js',
'resources/assets/vendor/libs/cleave-zen/cleave-zen.js',
'resources/assets/vendor/libs/jquery-repeater/jquery-repeater.js',
'resources/assets/js/offcanvas-add-payment.js'
])
@endsection

@section('page-script')
@vite('resources/assets/js/app-invoice-edit.js')
<script>
  'use strict';
  let invoiceItems = @json($invoice->rental->rentalItems);
  let currentInvoiceId = {{ $invoice->id }};

  document.addEventListener('DOMContentLoaded', function() {
    // Initialize repeater for invoice items
    $('.source-item').repeater({
      show: function() {
        $(this).slideDown();
      },
      hide: function(deleteElement) {
        if (confirm('Are you sure you want to delete this item?')) {
          $(this).slideUp(deleteElement);
          calculateTotal();
        }
      },
      ready: function(setIndexes) {
        // Load existing items
        invoiceItems.forEach((item, index) => {
          if (index > 0) {
            $('[data-repeater-create]').click();
          }
          const itemWrapper = $('[data-repeater-item]').eq(index);
          itemWrapper.find('.item-details').val(item.equipment.name).trigger('change');
          itemWrapper.find('.invoice-item-price').val(item.daily_rate);
          itemWrapper.find('.invoice-item-qty').val(item.quantity);
          itemWrapper.find('.invoice-item-days').val(item.days);
          calculateItemTotal(itemWrapper);
        });
        calculateTotal();
      }
    });

    function calculateItemTotal(itemWrapper) {
      const qty = parseFloat(itemWrapper.find('.invoice-item-qty').val()) || 0;
      const days = parseFloat(itemWrapper.find('.invoice-item-days').val()) || 0;
      const rate = parseFloat(itemWrapper.find('.invoice-item-price').val()) || 0;
      const subtotal = qty * days * rate;
      itemWrapper.find('.item-subtotal').text(formatCurrency(subtotal));
      calculateTotal();
    }

    function calculateTotal() {
      let subtotal = 0;
      $('[data-repeater-item]').each(function() {
        const qty = parseFloat($(this).find('.invoice-item-qty').val()) || 0;
        const days = parseFloat($(this).find('.invoice-item-days').val()) || 0;
        const rate = parseFloat($(this).find('.invoice-item-price').val()) || 0;
        subtotal += qty * days * rate;
      });

      const discount = parseFloat($('#discount').val()) || 0;
      const discountAmount = subtotal * (discount / 100);
      const grandTotal = subtotal - discountAmount;

      $('.subtotal-amount').text(formatCurrency(subtotal));
      $('.discount-amount').text(formatCurrency(discountAmount));
      $('.total-amount').text(formatCurrency(grandTotal));
    }

    function formatCurrency(amount) {
      return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: 'LKR',
        minimumFractionDigits: 2
      }).format(amount);
    }

    // Handle discount change
    $('#discount').on('input', calculateTotal);

    // Handle quantity and days change
    $(document).on('input', '.invoice-item-qty, .invoice-item-days', function() {
      calculateItemTotal($(this).closest('[data-repeater-item]'));
    });

    // Print function
    $('#printBtn').on('click', function() {
      window.print();
    });

    // Download PDF
    $('#downloadBtn').on('click', function() {
      window.location.href = '{{ route("invoices.download", $invoice) }}';
    });

    // Payment form submission
    $('#paymentForm').on('submit', function(e) {
      e.preventDefault();
      
      const formData = {
        invoice_id: $('#invoice-id').val(),
        amount: parseFloat($('#invoiceAmount').val()),
        method: $('#payment-method').val(),
        paid_at: $('#payment-date').val(),
        notes: $('#payment-note').val() || null,
      };

      fetch('/api/payments', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        body: JSON.stringify(formData)
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          // Close offcanvas
          const offcanvasElement = document.getElementById('addPaymentOffcanvas');
          const offcanvas = bootstrap.Offcanvas.getInstance(offcanvasElement);
          if (offcanvas) {
            offcanvas.hide();
          }
          
          // Reload page to show updated payment status
          window.location.reload();
        } else {
          alert('Error: ' + (data.message || 'Failed to add payment'));
        }
      })
      .catch(error => {
        console.error('Error:', error);
        alert('An error occurred while adding payment');
      });
    });
  });
</script>
@endsection

@section('content')
<div class="row invoice-edit">
  <!-- Invoice Edit-->
  <div class="col-lg-9 col-12 mb-lg-0 mb-6">
    <div class="card invoice-preview-card p-sm-12 p-6">
      <div class="card-body invoice-preview-header rounded p-6 px-3 text-heading">
        <div class="row mx-0 px-3">
          <div class="col-md-7 mb-md-0 mb-6 ps-0">
            <div class="d-flex svg-illustration align-items-center gap-3 mb-6">
              <span class="app-brand-logo demo">@include('_partials.macros')</span>
              <span class="mb-0 app-brand-text fw-semibold">{{ config('variables.templateName') }}</span>
            </div>
            <p class="mb-1">{{ $invoice->rental->shop->address ?? 'Equipment Renting System' }}</p>
            <p class="mb-1">Phone: {{ $invoice->rental->shop->phone ?? 'N/A' }}</p>
          </div>
          <div class="col-md-5 col-8 pe-0 ps-0 ps-md-2">
            <dl class="row mb-0 gx-4">
              <dt class="col-sm-5 mb-2 d-md-flex align-items-center justify-content-start">
                <span class="h5 text-capitalize mb-0 text-nowrap">Invoice</span>
              </dt>
              <dd class="col-sm-7">
                <div class="input-group input-group-sm input-group-merge disabled">
                  <span class="input-group-text">#</span>
                  <input type="text" class="form-control" disabled value="{{ $invoice->invoice_number }}" />
                </div>
              </dd>
              <dt class="col-sm-5 mb-2 d-md-flex align-items-center justify-content-start">
                <span class="fw-normal text-nowrap">Date Issued:</span>
              </dt>
              <dd class="col-sm-7">
                <input type="text" class="form-control form-control-sm" disabled value="{{ $invoice->issued_at->format('Y-m-d') }}" />
              </dd>
              <dt class="col-sm-5 d-md-flex align-items-center justify-content-start">
                <span class="fw-normal">Due Date:</span>
              </dt>
              <dd class="col-sm-7 mb-0">
                <input type="text" class="form-control form-control-sm" disabled value="{{ $invoice->due_date?->format('Y-m-d') ?? ($invoice->rental->expected_return_date?->format('Y-m-d') ?? 'N/A') }}" />
                @if($invoice->isOverdue())
                  <span class="badge bg-danger mt-1">OVERDUE</span>
                @endif
              </dd>
            </dl>
          </div>
        </div>
      </div>

      <div class="card-body px-0">
        <div class="row my-1">
          <div class="col-md-6 col-sm-5 col-12 mb-sm-0 mb-6">
            <h6>Invoice To:</h6>
            <p class="mb-1"><strong>{{ $invoice->rental->customer?->name ?? 'Walk-in Customer' }}</strong></p>
            <p class="mb-1">{{ $invoice->rental->customer?->address ?? 'N/A' }}</p>
            <p class="mb-1">{{ $invoice->rental->customer?->phone ?? 'N/A' }}</p>
            @if($invoice->rental->customer?->nic_number)
              <p class="mb-0">NIC: {{ $invoice->rental->customer->nic_number }}</p>
            @endif
          </div>
          <div class="col-md-6 col-sm-7">
            <h6>Bill To:</h6>
            <table>
              <tbody>
                <tr>
                  <td class="pe-4">Total Due:</td>
                  <td>{{ number_format($invoice->amount, 2) }} LKR</td>
                </tr>
                <tr>
                  <td class="pe-4">Status:</td>
                  <td>
                    <span class="badge bg-{{ $invoice->status === 'paid' ? 'success' : ($invoice->status === 'partially_paid' ? 'warning' : 'danger') }}">
                      {{ ucfirst(str_replace('_', ' ', $invoice->status)) }}
                    </span>
                  </td>
                </tr>
                <tr>
                  <td class="pe-4">Rental Date:</td>
                  <td>{{ $invoice->rental->rental_date->format('Y-m-d H:i') }}</td>
                </tr>
                @if($invoice->rental->actual_return_date)
                <tr>
                  <td class="pe-4">Returned:</td>
                  <td>{{ $invoice->rental->actual_return_date->format('Y-m-d H:i') }}</td>
                </tr>
                @endif
              </tbody>
            </table>
          </div>
        </div>
      </div>
      <hr class="mb-6 mt-1" />
      <div class="card-body pt-0 px-0">
        <form class="source-item">
          <div class="mb-4" data-repeater-list="group-a">
            @foreach($invoice->rental->rentalItems as $index => $item)
            <div class="repeater-wrapper pt-0 pt-md-9" data-repeater-item>
              <div class="d-flex border rounded position-relative pe-0">
                <div class="row w-100 p-5 gx-5">
                  <div class="col-md-7 col-12 mb-md-0 mb-3">
                    <h6 class="mb-2 repeater-title fw-medium">Item</h6>
                    <input type="text" class="form-control item-details mb-4" value="{{ $item->equipment->name }}" readonly />
                    <textarea class="form-control" rows="2" readonly>{{ $item->equipment->description ?? 'N/A' }}</textarea>
                  </div>
                  <div class="col-md-2 col-12 mb-md-0 mb-5">
                    <h6 class="h6 repeater-title">Rate/Day</h6>
                    <input type="text" class="form-control invoice-item-price mb-5" value="{{ number_format($item->daily_rate, 2) }}" readonly />
                  </div>
                  <div class="col-md-2 col-12 mb-md-0 mb-4">
                    <h6 class="h6 repeater-title">Qty</h6>
                    <input type="text" class="form-control invoice-item-qty" value="{{ $item->quantity }}" readonly />
                  </div>
                  <div class="col-md-2 col-12 mb-md-0 mb-4">
                    <h6 class="h6 repeater-title">Days</h6>
                    <input type="text" class="form-control invoice-item-days" value="{{ number_format($item->days, 2) }}" readonly />
                  </div>
                  <div class="col-md-1 col-12 pe-0">
                    <h6 class="h6 repeater-title">Price</h6>
                    <p class="mb-0 mt-2 text-heading item-subtotal">{{ number_format($item->subtotal, 2) }} LKR</p>
                  </div>
                </div>
              </div>
            </div>
            @endforeach
          </div>
        </form>
      </div>
      <hr class="my-1" />
      <div class="card-body px-0">
        <div class="row">
          <div class="col-md-6 mb-md-0 mb-3">
            @if($invoice->rental->remarks)
            <div class="d-flex align-items-center mb-4">
              <label class="me-2 h6 mb-2">Remarks:</label>
            </div>
            <p>{{ $invoice->rental->remarks }}</p>
            @endif
          </div>
          <div class="col-md-6 d-flex justify-content-md-end mt-2">
            <div class="invoice-calculations">
              <div class="d-flex justify-content-between mb-1">
                <span class="w-px-100">Subtotal:</span>
                <h6 class="mb-0 subtotal-amount">{{ number_format($invoice->rental->total_amount, 2) }} LKR</h6>
              </div>
              @if($invoice->rental->discount > 0)
              <div class="d-flex justify-content-between mb-1">
                <span class="w-px-100">Discount:</span>
                <h6 class="mb-0 discount-amount">{{ number_format($invoice->rental->discount, 2) }} LKR</h6>
              </div>
              @endif
              <hr class="my-2" />
              <div class="d-flex justify-content-between">
                <span class="w-px-100">Total:</span>
                <h6 class="mb-0 total-amount">{{ number_format($invoice->rental->grand_total, 2) }} LKR</h6>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
  <!-- /Invoice Edit-->

  <!-- Invoice Actions -->
  <div class="col-lg-3 col-12 invoice-actions no-print">
    <div class="card mb-6">
      <div class="card-body">
        <button type="button" id="printBtn" class="btn btn-primary d-grid w-100 mb-4">
          <span class="d-flex align-items-center justify-content-center text-nowrap">
            <i class="icon-base ri ri-printer-line icon-16px scaleX-n1-rtl me-2"></i>Print A4 Invoice</span>
        </button>
        <a href="{{ route('invoices.thermal', $invoice) }}" target="_blank" class="btn btn-outline-primary d-grid w-100 mb-4">
          <span class="d-flex align-items-center justify-content-center text-nowrap">
            <i class="icon-base ri ri-printer-line icon-16px scaleX-n1-rtl me-2"></i>Print Thermal Receipt</span>
        </a>
        <div class="d-flex">
          <a href="{{ route('invoices.download', $invoice) }}" id="downloadBtn" class="btn btn-outline-secondary w-100 me-4 mb-4">Download PDF</a>
        </div>
        <a href="{{ route('rentals') }}" class="btn btn-outline-secondary d-grid w-100 mb-4">Back to Rentals</a>
        @if($invoice->canBeEdited())
        <a href="{{ route('invoices.edit', $invoice) }}" class="btn btn-warning d-grid w-100 mb-4">
          <span class="d-flex align-items-center justify-content-center text-nowrap">
            <i class="icon-base ri ri-edit-line icon-16px scaleX-n1-rtl me-2"></i>Edit Invoice</span>
        </a>
        @else
        <button type="button" class="btn btn-outline-secondary d-grid w-100 mb-4" disabled>
          <span class="d-flex align-items-center justify-content-center text-nowrap">
            <i class="icon-base ri ri-edit-line icon-16px scaleX-n1-rtl me-2"></i>Edit Invoice (Payment Processed)</span>
        </button>
        @endif
        @if($invoice->status !== 'paid')
        <button type="button" class="btn btn-success d-grid w-100" data-bs-toggle="offcanvas" data-bs-target="#addPaymentOffcanvas">
          <span class="d-flex align-items-center justify-content-center text-nowrap">
            <i class="icon-base ri ri-money-dollar-circle-line icon-16px scaleX-n1-rtl me-2"></i>Add Payment</span>
        </button>
        @endif
      </div>
    </div>
  </div>
  <!-- /Invoice Actions -->
</div>

<!-- Offcanvas -->
@include('_partials/_offcanvas/offcanvas-add-payment')
<!-- /Offcanvas -->
@endsection
