@php
  $configData = Helper::appClasses();
  $revenueGrowth = $lastMonthRevenue > 0 ? (($thisMonthRevenue - $lastMonthRevenue) / $lastMonthRevenue) * 100 : 0;
  $rentalGrowth = $lastMonthRentals > 0 ? (($thisMonthRentals - $lastMonthRentals) / $lastMonthRentals) * 100 : 0;
@endphp
@extends('layouts/layoutMaster')

@section('title', 'Dashboard - Equipment Rental System')
@section('vendor-style')
  @vite(['resources/assets/vendor/libs/apex-charts/apex-charts.scss'])
@endsection

@section('page-style')
  @vite(['resources/assets/vendor/scss/pages/cards-statistics.scss'])
@endsection

@section('vendor-script')
  @vite(['resources/assets/vendor/libs/apex-charts/apexcharts.js'])
@endsection

@section('page-script')
  <script>
    (function() {
      'use strict';

      // Wait for DOM and ApexCharts to be ready
      function initCharts() {
        if (typeof ApexCharts === 'undefined') {
          setTimeout(initCharts, 100);
          return;
        }

        // Monthly Revenue Chart
        @if(auth()->user()->hasPermission('dashboard.revenue_chart'))
        const monthlyRevenueData = @json($monthlyRevenue);
        const monthlyRevenueChartEl = document.querySelector('#monthlyRevenueChart');
        if (monthlyRevenueChartEl) {
          const revenueChartOptions = {
            chart: {
              type: 'area',
              height: 300,
              toolbar: {
                show: false
              }
            },
            series: [{
              name: 'Revenue',
              data: monthlyRevenueData.map(item => parseFloat(item.revenue) || 0)
            }],
            xaxis: {
              categories: monthlyRevenueData.map(item => item.month)
            },
            colors: ['#696cff'],
            fill: {
              type: 'gradient',
              gradient: {
                shadeIntensity: 1,
                opacityFrom: 0.7,
                opacityTo: 0.3
              }
            }
          };
          const revenueChart = new ApexCharts(monthlyRevenueChartEl, revenueChartOptions);
          revenueChart.render();
        }
        @endif

        // Status Distribution Chart
        @if(auth()->user()->hasPermission('dashboard.status_chart'))
        const statusData = @json($statusDistribution);
        const statusChartEl = document.querySelector('#statusChart');
        if (statusChartEl && Object.keys(statusData).length > 0) {
          const statusChartOptions = {
            chart: {
              type: 'donut',
              height: 300
            },
            series: Object.values(statusData),
            labels: Object.keys(statusData).map(key => key.charAt(0).toUpperCase() + key.slice(1)),
            colors: ['#696cff', '#71dd37', '#ffab00', '#ff3e1d'],
            legend: {
              position: 'bottom'
            }
          };
          const statusChart = new ApexCharts(statusChartEl, statusChartOptions);
          statusChart.render();
        }
        @endif

        // Equipment Utilization Chart
        @if(auth()->user()->hasPermission('dashboard.equipment_chart'))
        const equipmentData = @json($equipmentUtilization);
        const equipmentChartEl = document.querySelector('#equipmentChart');
        if (equipmentChartEl && equipmentData.length > 0) {
          const equipmentChartOptions = {
            chart: {
              type: 'bar',
              height: 300,
              toolbar: {
                show: false
              }
            },
            series: [{
              name: 'Times Rented',
              data: equipmentData.map(item => parseInt(item.total_rented) || 0)
            }],
            xaxis: {
              categories: equipmentData.map(item => item.name)
            },
            colors: ['#696cff'],
            plotOptions: {
              bar: {
                horizontal: true
              }
            }
          };
          const equipmentChart = new ApexCharts(equipmentChartEl, equipmentChartOptions);
          equipmentChart.render();
        }
        @endif

        // Daily Rental Trends Chart
        @if(auth()->user()->hasPermission('dashboard.daily_rentals_chart'))
        const dailyRentalsData = @json($dailyRentals);
        const dailyRentalsChartEl = document.querySelector('#dailyRentalsChart');
        if (dailyRentalsChartEl && dailyRentalsData.length > 0) {
          const dailyRentalsChartOptions = {
            chart: {
              type: 'line',
              height: 300,
              toolbar: {
                show: false
              }
            },
            series: [{
                name: 'Rentals',
                type: 'column',
                data: dailyRentalsData.map(item => item.count)
              },
              {
                name: 'Revenue (Rs.)',
                type: 'line',
                data: dailyRentalsData.map(item => parseFloat(item.revenue) || 0)
              }
            ],
            xaxis: {
              categories: dailyRentalsData.map(item => item.date),
              labels: {
                rotate: -45,
                rotateAlways: false
              }
            },
            yaxis: [{
                title: {
                  text: 'Number of Rentals'
                },
                labels: {
                  formatter: function(val) {
                    return Math.round(val);
                  }
                }
              },
              {
                opposite: true,
                title: {
                  text: 'Revenue (Rs.)'
                },
                labels: {
                  formatter: function(val) {
                    return 'Rs. ' + Math.round(val).toLocaleString();
                  }
                }
              }
            ],
            colors: ['#696cff', '#71dd37'],
            stroke: {
              width: [0, 3],
              curve: ['straight', 'smooth']
            },
            fill: {
              opacity: [0.8, 1]
            },
            plotOptions: {
              bar: {
                columnWidth: '60%'
              }
            },
            legend: {
              position: 'top'
            },
            tooltip: {
              shared: true,
              intersect: false,
              y: {
                formatter: function(val, {
                  seriesIndex
                }) {
                  if (seriesIndex === 0) {
                    return val + ' rentals';
                  } else {
                    return 'Rs. ' + val.toLocaleString('en-US', {
                      minimumFractionDigits: 2,
                      maximumFractionDigits: 2
                    });
                  }
                }
              }
            }
          };
          const dailyRentalsChart = new ApexCharts(dailyRentalsChartEl, dailyRentalsChartOptions);
          dailyRentalsChart.render();
        }
        @endif
      }

      // Initialize when DOM is ready
      if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initCharts);
      } else {
        initCharts();
      }
    })();
  </script>
@endsection

@section('content')
  <div class="row g-6">
    <!-- Welcome Card -->
    <div class="col-12">
      <div class="card">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-center flex-wrap">
            <div>
              <h4 class="card-title mb-2">Welcome back, <span class="fw-bold">{{ auth()->user()->name }}!</span> 👋</h4>
              <p class="mb-0">Here's what's happening with your equipment rental business today.</p>
            </div>
            @if(auth()->user()->hasPermission('rentals.create'))
            <div>
              <a href="{{ route('rentals.create') }}" class="btn btn-primary">
                <i class="icon-base ri ri-add-line me-1"></i>New Rental
              </a>
            </div>
            @endif
          </div>
        </div>
      </div>
    </div>
    <!--/ Welcome Card -->

    <!-- Statistics Cards -->
    <!-- Total Rentals -->
    @if(auth()->user()->hasPermission('dashboard.total_rentals'))
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-primary rounded-3">
                <i class="icon-base ri ri-shopping-cart-2-line icon-24px"></i>
              </div>
            </div>
            <div class="d-flex align-items-center">
              @if ($rentalGrowth >= 0)
                <p class="mb-0 text-success me-1">{{ number_format($rentalGrowth, 1) }}%</p>
                <i class="icon-base ri ri-arrow-up-s-line text-success"></i>
              @else
                <p class="mb-0 text-danger me-1">{{ number_format(abs($rentalGrowth), 1) }}%</p>
                <i class="icon-base ri ri-arrow-down-s-line text-danger"></i>
              @endif
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">{{ number_format($totalRentals) }}</h5>
            <p class="mb-0">Total Rentals</p>
            <div class="badge bg-label-secondary rounded-pill mt-2">{{ $thisMonthRentals }} this month</div>
          </div>
        </div>
      </div>
    </div>
    @endif
    <!--/ Total Rentals -->

    <!-- Active Rentals -->
    @if(auth()->user()->hasPermission('dashboard.active_rentals'))
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-warning rounded-3">
                <i class="icon-base ri ri-time-line icon-24px"></i>
              </div>
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">{{ number_format($activeRentals) }}</h5>
            <p class="mb-0">Active Rentals</p>
            <div class="badge bg-label-warning rounded-pill mt-2">{{ $returnedRentals }} returned</div>
          </div>
        </div>
      </div>
    </div>
    @endif
    <!--/ Active Rentals -->

    <!-- Total Revenue -->
    @if(auth()->user()->hasPermission('dashboard.total_revenue'))
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-success rounded-3">
                <i class="icon-base ri ri-money-dollar-circle-line icon-24px"></i>
              </div>
            </div>
            <div class="d-flex align-items-center">
              @if ($revenueGrowth >= 0)
                <p class="mb-0 text-success me-1">{{ number_format($revenueGrowth, 1) }}%</p>
                <i class="icon-base ri ri-arrow-up-s-line text-success"></i>
              @else
                <p class="mb-0 text-danger me-1">{{ number_format(abs($revenueGrowth), 1) }}%</p>
                <i class="icon-base ri ri-arrow-down-s-line text-danger"></i>
              @endif
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">Rs. {{ number_format($totalRevenue, 2) }}</h5>
            <p class="mb-0">Total Revenue</p>
            <div class="badge bg-label-secondary rounded-pill mt-2">Rs. {{ number_format($thisMonthRevenue, 2) }} this
              month</div>
          </div>
        </div>
      </div>
    </div>
    @endif
    <!--/ Total Revenue -->

    <!-- Today's Revenue -->
    @if(auth()->user()->hasPermission('dashboard.today_revenue'))
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-info rounded-3">
                <i class="icon-base ri ri-calendar-line icon-24px"></i>
              </div>
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">Rs. {{ number_format($todayRevenue, 2) }}</h5>
            <p class="mb-0">Today's Revenue</p>
            <div class="badge bg-label-info rounded-pill mt-2">{{ $todayRentals }} rentals today</div>
          </div>
        </div>
      </div>
    </div>
    @endif
    <!--/ Today's Revenue -->

    <!-- Quick Stats Row -->
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-danger rounded-3">
                <i class="icon-base ri ri-file-list-3-line icon-24px"></i>
              </div>
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">{{ number_format($unpaidInvoices) }}</h5>
            <p class="mb-0">Unpaid Invoices</p>
            <div class="badge bg-label-danger rounded-pill mt-2">Rs. {{ number_format($unpaidAmount, 2) }}</div>
          </div>
        </div>
      </div>
    </div>

    @if(auth()->user()->hasPermission('dashboard.total_shops') || auth()->user()->hasPermission('shops.view'))
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-primary rounded-3">
                <i class="icon-base ri ri-store-line icon-24px"></i>
              </div>
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">{{ number_format($totalShops) }}</h5>
            <p class="mb-0">Total Shops</p>
            <div class="badge bg-label-secondary rounded-pill mt-2">{{ $isAdmin ? 'All locations' : 'Your shop' }}</div>
          </div>
        </div>
      </div>
    </div>
    @endif

    @if(auth()->user()->hasPermission('dashboard.total_equipment'))
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-success rounded-3">
                <i class="icon-base ri ri-tools-line icon-24px"></i>
              </div>
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">{{ number_format($totalEquipments) }}</h5>
            <p class="mb-0">Total Equipment</p>
            <div class="badge bg-label-secondary rounded-pill mt-2">In catalog</div>
          </div>
        </div>
      </div>
    </div>
    @endif

    @if(auth()->user()->hasPermission('dashboard.total_customers'))
    <div class="col-xxl-3 col-md-6 col-sm-6">
      <div class="card h-100">
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start flex-wrap gap-2">
            <div class="avatar">
              <div class="avatar-initial bg-label-info rounded-3">
                <i class="icon-base ri ri-user-line icon-24px"></i>
              </div>
            </div>
          </div>
          <div class="card-info mt-5">
            <h5 class="mb-1">{{ number_format($totalCustomers) }}</h5>
            <p class="mb-0">Total Customers</p>
            <div class="badge bg-label-secondary rounded-pill mt-2">Registered</div>
          </div>
        </div>
      </div>
    </div>
    @endif

    <!-- Monthly Revenue Chart -->
    @if(auth()->user()->hasPermission('dashboard.revenue_chart'))
    <div class="col-12 col-xxl-8">
      <div class="card h-100">
        <div class="card-header">
          <div class="d-flex justify-content-between">
            <h5 class="mb-0">Monthly Revenue Trend</h5>
            <div class="dropdown">
              <button class="btn btn-text-secondary rounded-pill text-body-secondary border-0 p-1" type="button"
                id="revenueDropdown" data-bs-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                <i class="icon-base ri ri-more-2-line"></i>
              </button>
              <div class="dropdown-menu dropdown-menu-end" aria-labelledby="revenueDropdown">
                <a class="dropdown-item" href="javascript:void(0);">Last 6 Months</a>
                <a class="dropdown-item" href="javascript:void(0);">Last Year</a>
              </div>
            </div>
          </div>
        </div>
        <div class="card-body">
          <div id="monthlyRevenueChart"></div>
        </div>
      </div>
    </div>
    @endif
    <!--/ Monthly Revenue Chart -->

    <!-- Status Distribution & Equipment Utilization -->
    @if(auth()->user()->hasPermission('dashboard.status_chart'))
    <div class="col-12 col-xxl-4">
      <div class="card h-100">
        <div class="card-header">
          <h5 class="mb-0">Rental Status</h5>
        </div>
        <div class="card-body">
          <div id="statusChart"></div>
        </div>
      </div>
    </div>
    @endif

    <!-- Stock Alerts -->
    @if(auth()->user()->hasPermission('dashboard.active_stock') && ($lowStockItems->count() > 0 || $outOfStockItems->count() > 0))
      <div class="col-12 col-xxl-6">
        <div class="card h-100">
          <div class="card-header">
            <div class="d-flex justify-content-between">
              <h5 class="mb-0">Stock Alerts</h5>
              <a href="{{ route('stocks') }}" class="btn btn-sm btn-primary">View All</a>
            </div>
          </div>
          <div class="card-body">
            @if ($outOfStockItems->count() > 0)
              <div class="alert alert-danger mb-3">
                <h6 class="alert-heading mb-2">
                  <i class="icon-base ri ri-error-warning-line me-1"></i>
                  Out of Stock ({{ $outOfStockItems->count() }})
                </h6>
                <ul class="mb-0 ps-3">
                  @foreach ($outOfStockItems->take(5) as $stock)
                    <li>
                      <strong>{{ $stock->equipment->name }}</strong>
                      @if ($isAdmin)
                        - {{ $stock->shop->name }}
                      @endif
                      <span class="badge bg-danger ms-2">0 available</span>
                    </li>
                  @endforeach
                </ul>
              </div>
            @endif

            @if ($lowStockItems->count() > 0)
              <div class="alert alert-warning mb-0">
                <h6 class="alert-heading mb-2">
                  <i class="icon-base ri ri-alert-line me-1"></i>
                  Low Stock ({{ $lowStockItems->count() }})
                </h6>
                <ul class="mb-0 ps-3">
                  @foreach ($lowStockItems->take(5) as $stock)
                    <li>
                      <strong>{{ $stock->equipment->name }}</strong>
                      @if ($isAdmin)
                        - {{ $stock->shop->name }}
                      @endif
                      <span class="badge bg-warning ms-2">{{ $stock->quantity - $stock->reserved }} available
                        (threshold: {{ $stock->minimum_threshold }})
                      </span>
                    </li>
                  @endforeach
                </ul>
              </div>
            @endif
          </div>
        </div>
      </div>
    @endif

    <!-- Equipment Utilization -->
    @if(auth()->user()->hasPermission('dashboard.equipment_chart') && $equipmentUtilization->count() > 0)
      <div class="col-12 col-xxl-6">
        <div class="card h-100">
          <div class="card-header">
            <h5 class="mb-0">Top Rented Equipment</h5>
          </div>
          <div class="card-body">
            <div id="equipmentChart"></div>
          </div>
        </div>
      </div>
    @endif

    <!-- Daily Rental Trends -->
    @if(auth()->user()->hasPermission('dashboard.daily_rentals_chart'))
    <div class="col-12 col-xxl-6">
      <div class="card h-100">
        <div class="card-header">
          <div class="d-flex justify-content-between">
            <h5 class="mb-0">Daily Rental Trends</h5>
            <div class="dropdown">
              <button class="btn btn-text-secondary rounded-pill text-body-secondary border-0 p-1" type="button"
                id="dailyTrendsDropdown" data-bs-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                <i class="icon-base ri ri-more-2-line"></i>
              </button>
              <div class="dropdown-menu dropdown-menu-end" aria-labelledby="dailyTrendsDropdown">
                <a class="dropdown-item" href="javascript:void(0);">Last 14 Days</a>
                <a class="dropdown-item" href="javascript:void(0);">Last 30 Days</a>
              </div>
            </div>
          </div>
          <p class="mb-0 card-subtitle px-4 pt-2">Rentals and revenue over the last 14 days</p>
        </div>
        <div class="card-body">
          <div id="dailyRentalsChart"></div>
        </div>
      </div>
    </div>
    @endif
    <!--/ Daily Rental Trends -->

    <!-- Recent Rentals -->
    @if(auth()->user()->hasPermission('rentals.view'))
    <div class="col-12 col-xxl-8">
      <div class="card h-100">
        <div class="card-header">
          <div class="d-flex justify-content-between">
            <h5 class="mb-0">Recent Rentals</h5>
            <a href="{{ route('rentals') }}" class="btn btn-sm btn-primary">View All</a>
          </div>
        </div>
        <div class="card-body">
          <div class="table-responsive">
            <table class="table table-hover">
              <thead>
                <tr>
                  <th>Rental ID</th>
                  <th>Customer</th>
                  @if ($isAdmin)
                    <th>Shop</th>
                  @endif
                  <th>Date</th>
                  <th>Amount</th>
                  <th>Status</th>
                </tr>
              </thead>
              <tbody>
                @forelse($recentRentals as $rental)
                  <tr>
                    <td>
                      <a href="{{ route('rentals') }}?id={{ $rental->id }}"
                        class="fw-medium">#{{ $rental->id }}</a>
                    </td>
                    <td>
                      {{ $rental->customer ? $rental->customer->name : 'Walk-in Customer' }}
                      @if ($rental->customer && $rental->customer->phone)
                        <br><small class="text-muted">{{ $rental->customer->phone }}</small>
                      @endif
                    </td>
                    @if ($isAdmin)
                      <td>{{ $rental->shop->name }}</td>
                    @endif
                    <td>{{ \Carbon\Carbon::parse($rental->rental_date)->format('M d, Y') }}</td>
                    <td class="fw-medium">Rs. {{ number_format($rental->grand_total, 2) }}</td>
                    <td>
                      @if ($rental->status == 'active')
                        <span class="badge bg-label-warning">Active</span>
                      @elseif($rental->status == 'returned')
                        <span class="badge bg-label-success">Returned</span>
                      @elseif($rental->status == 'cancelled')
                        <span class="badge bg-label-danger">Cancelled</span>
                      @else
                        <span class="badge bg-label-secondary">{{ ucfirst($rental->status) }}</span>
                      @endif
                    </td>
                  </tr>
                @empty
                  <tr>
                    <td colspan="{{ $isAdmin ? '6' : '5' }}" class="text-center text-muted py-4">
                      No rentals found
                    </td>
                  </tr>
                @endforelse
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </div>
    @endif
    <!--/ Recent Rentals -->

    <!-- Shop Performance (Admin Only) -->
    @if (auth()->user()->hasPermission('shops.view') && $isAdmin && $shopPerformance->count() > 0)
      <div class="col-12 col-xxl-4">
        <div class="card h-100">
          <div class="card-header">
            <h5 class="mb-0">Shop Performance (This Month)</h5>
          </div>
          <div class="card-body">
            <div class="d-flex justify-content-between p-3 border-bottom">
              <p class="mb-0 fw-medium">SHOP</p>
              <p class="mb-0 fw-medium">REVENUE</p>
            </div>
            <ul class="p-0 m-0">
              @foreach ($shopPerformance as $shop)
                <li class="d-flex align-items-center justify-content-between p-3 border-bottom">
                  <div>
                    <h6 class="mb-1">{{ $shop->name }}</h6>
                    <small class="text-muted">{{ $shop->rentals_count ?? 0 }} rentals</small>
                  </div>
                  <div class="text-end">
                    <h6 class="mb-0">Rs. {{ number_format($shop->rentals_sum_grand_total ?? 0, 2) }}</h6>
                  </div>
                </li>
              @endforeach
            </ul>
          </div>
        </div>
      </div>
    @endif
    <!--/ Shop Performance -->
  </div>
@endsection
