<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // For SQLite, we need to recreate the tables
        if (DB::getDriverName() === 'sqlite') {
            // Handle equipments table
            DB::statement('PRAGMA foreign_keys=OFF;');
            
            // Rename hourly_rate to daily_rate in equipments
            DB::statement('
                CREATE TABLE equipments_new (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    sku TEXT,
                    name TEXT NOT NULL,
                    category_id INTEGER,
                    description TEXT,
                    daily_rate NUMERIC NOT NULL DEFAULT 0,
                    replacement_cost NUMERIC NOT NULL DEFAULT 0,
                    is_serial_tracked INTEGER NOT NULL DEFAULT 0,
                    created_at DATETIME,
                    updated_at DATETIME,
                    deleted_at DATETIME,
                    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
                )
            ');
            
            DB::statement('INSERT INTO equipments_new (id, sku, name, category_id, description, daily_rate, replacement_cost, is_serial_tracked, created_at, updated_at, deleted_at) 
                           SELECT id, sku, name, category_id, description, hourly_rate, replacement_cost, is_serial_tracked, created_at, updated_at, deleted_at FROM equipments');
            
            DB::statement('DROP TABLE equipments');
            DB::statement('ALTER TABLE equipments_new RENAME TO equipments');
            
            // Create indexes
            DB::statement('CREATE UNIQUE INDEX equipments_sku_unique ON equipments(sku) WHERE sku IS NOT NULL');
            DB::statement('CREATE INDEX equipments_category_id_index ON equipments(category_id)');
            
            // Handle rental_items table - rename hourly_rate to daily_rate and hours to days
            DB::statement('
                CREATE TABLE rental_items_new (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    rental_id INTEGER NOT NULL,
                    equipment_id INTEGER NOT NULL,
                    quantity INTEGER NOT NULL,
                    daily_rate NUMERIC NOT NULL,
                    days NUMERIC NOT NULL,
                    subtotal NUMERIC NOT NULL,
                    serial_numbers TEXT,
                    created_at DATETIME,
                    updated_at DATETIME,
                    FOREIGN KEY (rental_id) REFERENCES rentals(id) ON DELETE CASCADE,
                    FOREIGN KEY (equipment_id) REFERENCES equipments(id) ON DELETE CASCADE
                )
            ');
            
            DB::statement('INSERT INTO rental_items_new (id, rental_id, equipment_id, quantity, daily_rate, days, subtotal, serial_numbers, created_at, updated_at) 
                           SELECT id, rental_id, equipment_id, quantity, hourly_rate, hours, subtotal, serial_numbers, created_at, updated_at FROM rental_items');
            
            DB::statement('DROP TABLE rental_items');
            DB::statement('ALTER TABLE rental_items_new RENAME TO rental_items');
            
            DB::statement('PRAGMA foreign_keys=ON;');
        } else {
            // For other databases (MySQL, PostgreSQL, etc.)
            // Rename hourly_rate to daily_rate in equipments table
            Schema::table('equipments', function (Blueprint $table) {
                $table->renameColumn('hourly_rate', 'daily_rate');
            });
            
            // Rename hourly_rate to daily_rate and hours to days in rental_items table
            Schema::table('rental_items', function (Blueprint $table) {
                $table->renameColumn('hourly_rate', 'daily_rate');
                $table->renameColumn('hours', 'days');
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        if (DB::getDriverName() === 'sqlite') {
            // Reverse the changes for SQLite
            DB::statement('PRAGMA foreign_keys=OFF;');
            
            // Revert equipments table
            DB::statement('
                CREATE TABLE equipments_new (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    sku TEXT,
                    name TEXT NOT NULL,
                    category_id INTEGER,
                    description TEXT,
                    hourly_rate NUMERIC NOT NULL DEFAULT 0,
                    replacement_cost NUMERIC NOT NULL DEFAULT 0,
                    is_serial_tracked INTEGER NOT NULL DEFAULT 0,
                    created_at DATETIME,
                    updated_at DATETIME,
                    deleted_at DATETIME,
                    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
                )
            ');
            
            DB::statement('INSERT INTO equipments_new (id, sku, name, category_id, description, hourly_rate, replacement_cost, is_serial_tracked, created_at, updated_at, deleted_at) 
                           SELECT id, sku, name, category_id, description, daily_rate, replacement_cost, is_serial_tracked, created_at, updated_at, deleted_at FROM equipments');
            
            DB::statement('DROP TABLE equipments');
            DB::statement('ALTER TABLE equipments_new RENAME TO equipments');
            
            DB::statement('CREATE UNIQUE INDEX equipments_sku_unique ON equipments(sku) WHERE sku IS NOT NULL');
            DB::statement('CREATE INDEX equipments_category_id_index ON equipments(category_id)');
            
            // Revert rental_items table
            DB::statement('
                CREATE TABLE rental_items_new (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    rental_id INTEGER NOT NULL,
                    equipment_id INTEGER NOT NULL,
                    quantity INTEGER NOT NULL,
                    hourly_rate NUMERIC NOT NULL,
                    hours NUMERIC NOT NULL,
                    subtotal NUMERIC NOT NULL,
                    serial_numbers TEXT,
                    created_at DATETIME,
                    updated_at DATETIME,
                    FOREIGN KEY (rental_id) REFERENCES rentals(id) ON DELETE CASCADE,
                    FOREIGN KEY (equipment_id) REFERENCES equipments(id) ON DELETE CASCADE
                )
            ');
            
            DB::statement('INSERT INTO rental_items_new (id, rental_id, equipment_id, quantity, hourly_rate, hours, subtotal, serial_numbers, created_at, updated_at) 
                           SELECT id, rental_id, equipment_id, quantity, daily_rate, days, subtotal, serial_numbers, created_at, updated_at FROM rental_items');
            
            DB::statement('DROP TABLE rental_items');
            DB::statement('ALTER TABLE rental_items_new RENAME TO rental_items');
            
            DB::statement('PRAGMA foreign_keys=ON;');
        } else {
            // For other databases
            Schema::table('equipments', function (Blueprint $table) {
                $table->renameColumn('daily_rate', 'hourly_rate');
            });
            
            Schema::table('rental_items', function (Blueprint $table) {
                $table->renameColumn('daily_rate', 'hourly_rate');
                $table->renameColumn('days', 'hours');
            });
        }
    }
};
