<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // For SQLite, we need to recreate the table with the correct foreign key
        if (DB::getDriverName() === 'sqlite') {
            // Drop the existing foreign key constraint by recreating the table
            Schema::table('rental_items', function (Blueprint $table) {
                // SQLite doesn't support dropping foreign keys directly
                // We need to recreate the table
            });

            // Recreate the table with correct foreign key
            DB::statement('PRAGMA foreign_keys=OFF;');

            // Create new table with correct foreign key
            DB::statement('
                CREATE TABLE rental_items_new (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    rental_id INTEGER NOT NULL,
                    equipment_id INTEGER NOT NULL,
                    quantity INTEGER NOT NULL,
                    hourly_rate NUMERIC NOT NULL,
                    hours NUMERIC NOT NULL,
                    subtotal NUMERIC NOT NULL,
                    serial_numbers TEXT,
                    created_at DATETIME,
                    updated_at DATETIME,
                    FOREIGN KEY (rental_id) REFERENCES rentals(id) ON DELETE CASCADE,
                    FOREIGN KEY (equipment_id) REFERENCES equipments(id) ON DELETE CASCADE
                )
            ');

            // Copy data
            DB::statement('INSERT INTO rental_items_new SELECT * FROM rental_items;');

            // Drop old table
            DB::statement('DROP TABLE rental_items;');

            // Rename new table
            DB::statement('ALTER TABLE rental_items_new RENAME TO rental_items;');

            DB::statement('PRAGMA foreign_keys=ON;');
        } else {
            // For other databases, drop and recreate the foreign key
            Schema::table('rental_items', function (Blueprint $table) {
                $table->dropForeign(['equipment_id']);
            });

            Schema::table('rental_items', function (Blueprint $table) {
                $table->foreign('equipment_id')->references('id')->on('equipments')->onDelete('cascade');
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Revert to original (though this might not be needed)
        if (DB::getDriverName() !== 'sqlite') {
            Schema::table('rental_items', function (Blueprint $table) {
                $table->dropForeign(['equipment_id']);
            });

            Schema::table('rental_items', function (Blueprint $table) {
                $table->foreign('equipment_id')->references('id')->on('equipment')->onDelete('cascade');
            });
        }
    }
};
