<?php

namespace App\Services;

use App\Models\Stock;
use App\Models\StockMovement;
use App\Models\Rental;
use Illuminate\Support\Facades\DB;

class StockService
{
    /**
     * Reserve stock for a rental
     */
    public static function reserveStock(Rental $rental): void
    {
        foreach ($rental->rentalItems as $item) {
            $stock = Stock::where('shop_id', $rental->shop_id)
                ->where('equipment_id', $item->equipment_id)
                ->first();

            if ($stock) {
                $stock->increment('reserved', $item->quantity);

                // Record stock movement
                StockMovement::create([
                    'shop_id' => $rental->shop_id,
                    'equipment_id' => $item->equipment_id,
                    'movement_type' => 'rental_reserve',
                    'quantity' => -$item->quantity,
                    'reference_type' => Rental::class,
                    'reference_id' => $rental->id,
                    'notes' => "Reserved for rental #{$rental->id}",
                    'created_by' => $rental->rented_by,
                ]);
            }
        }
    }

    /**
     * Release reserved stock when rental is returned or cancelled
     */
    public static function releaseStock(Rental $rental): void
    {
        foreach ($rental->rentalItems as $item) {
            $stock = Stock::where('shop_id', $rental->shop_id)
                ->where('equipment_id', $item->equipment_id)
                ->first();

            if ($stock && $stock->reserved >= $item->quantity) {
                $stock->decrement('reserved', $item->quantity);

                // Record stock movement
                StockMovement::create([
                    'shop_id' => $rental->shop_id,
                    'equipment_id' => $item->equipment_id,
                    'movement_type' => 'return',
                    'quantity' => $item->quantity,
                    'reference_type' => Rental::class,
                    'reference_id' => $rental->id,
                    'notes' => "Returned from rental #{$rental->id}",
                    'created_by' => auth()->id() ?? $rental->rented_by,
                ]);
            }
        }
    }

    /**
     * Check stock availability
     */
    public static function checkAvailability(int $shopId, int $equipmentId, int $quantity): bool
    {
        $stock = Stock::where('shop_id', $shopId)
            ->where('equipment_id', $equipmentId)
            ->first();

        if (!$stock) {
            return false;
        }

        $available = $stock->quantity - $stock->reserved;
        return $available >= $quantity;
    }

    /**
     * Get available quantity
     */
    public static function getAvailable(int $shopId, int $equipmentId): int
    {
        $stock = Stock::where('shop_id', $shopId)
            ->where('equipment_id', $equipmentId)
            ->first();

        if (!$stock) {
            return 0;
        }

        return $stock->quantity - $stock->reserved;
    }

    /**
     * Add stock (incoming)
     */
    public static function addStock(int $shopId, int $equipmentId, int $quantity, ?string $notes = null): void
    {
        DB::transaction(function () use ($shopId, $equipmentId, $quantity, $notes) {
            $stock = Stock::firstOrCreate(
                ['shop_id' => $shopId, 'equipment_id' => $equipmentId],
                ['quantity' => 0, 'reserved' => 0]
            );

            $stock->increment('quantity', $quantity);

            StockMovement::create([
                'shop_id' => $shopId,
                'equipment_id' => $equipmentId,
                'movement_type' => 'in',
                'quantity' => $quantity,
                'notes' => $notes ?? 'Stock added',
                'created_by' => auth()->id() ?? 1,
            ]);
        });
    }

    /**
     * Adjust stock (manual adjustment)
     */
    public static function adjustStock(int $shopId, int $equipmentId, int $quantity, ?string $notes = null): void
    {
        DB::transaction(function () use ($shopId, $equipmentId, $quantity, $notes) {
            $stock = Stock::firstOrCreate(
                ['shop_id' => $shopId, 'equipment_id' => $equipmentId],
                ['quantity' => 0, 'reserved' => 0]
            );

            if ($quantity > 0) {
                $stock->increment('quantity', $quantity);
            } else {
                $stock->decrement('quantity', abs($quantity));
            }

            StockMovement::create([
                'shop_id' => $shopId,
                'equipment_id' => $equipmentId,
                'movement_type' => 'adjustment',
                'quantity' => $quantity,
                'notes' => $notes ?? 'Stock adjustment',
                'created_by' => auth()->id() ?? 1,
            ]);
        });
    }
}
