<?php

namespace App\Providers;

use Illuminate\Support\Facades\View;
use Illuminate\Routing\Route;

use Illuminate\Support\ServiceProvider;

class MenuServiceProvider extends ServiceProvider
{
  /**
   * Register services.
   */
  public function register(): void
  {
    //
  }

  /**
   * Bootstrap services.
   */
  public function boot(): void
  {
    // Use view composer to filter menu after authentication is established
    // This runs when views are rendered, ensuring auth is available
    view()->composer('*', function ($view) {
      static $menuDataCache = null;
      
      // Only process once per request
      if ($menuDataCache === null) {
        $verticalMenuJson = file_get_contents(base_path('resources/menu/verticalMenu.json'));
        $verticalMenuData = json_decode($verticalMenuJson);
        $horizontalMenuJson = file_get_contents(base_path('resources/menu/horizontalMenu.json'));
        $horizontalMenuData = json_decode($horizontalMenuJson);

        // Filter menu items based on user permissions
        $verticalMenuData = $this->filterMenuItems($verticalMenuData);
        $horizontalMenuData = $this->filterMenuItems($horizontalMenuData);

        $menuDataCache = [$verticalMenuData, $horizontalMenuData];
      }

      // Share menuData to the view if not already set
      if (!isset($view->getData()['menuData'])) {
        $view->with('menuData', $menuDataCache);
      }
    });
  }

  /**
   * Filter menu items based on user permissions
   */
  private function filterMenuItems($menuData)
  {
    if (!isset($menuData->menu)) {
      return $menuData;
    }

    // Use auth()->check() first to ensure session is available
    if (!auth()->check()) {
      return $menuData;
    }

    $user = auth()->user();

    // Double check user exists
    if (!$user) {
      return $menuData;
    }

    // Check if user is admin
    $isAdmin = false;
    if (method_exists($user, 'isAdmin')) {
      try {
        $isAdmin = $user->isAdmin();
      } catch (\Exception $e) {
        // If isAdmin check fails, assume not admin
        $isAdmin = false;
      }
    }

    $filteredMenu = array_filter($menuData->menu, function ($menu) use ($isAdmin, $user) {
      // Check if this is the Dashboard menu item
      $isDashboardMenu = false;
      if (isset($menu->slug) && $menu->slug === 'dashboard' && !isset($menu->submenu)) {
        $isDashboardMenu = true;
      }
      if (isset($menu->url) && ($menu->url === '/' || $menu->url === '/dashboard')) {
        $isDashboardMenu = true;
      }

      // Only show dashboard menu if user has dashboard.view permission
      if ($isDashboardMenu) {
        if ($isAdmin) {
          return true; // Admin has all permissions
        }
        if (method_exists($user, 'hasPermission')) {
          try {
            $hasPermission = $user->hasPermission('dashboard.view');
            return $hasPermission;
          } catch (\Exception $e) {
            // If permission check fails, deny access
            return false;
          }
        }
        return false;
      }

      // Filter submenu items
      if (isset($menu->submenu) && is_array($menu->submenu)) {
        $menu->submenu = array_filter($menu->submenu, function ($submenu) use ($isAdmin) {
          // Check if this is the permissions submenu
          $isPermissionsMenu = false;
          if (isset($submenu->url) && str_contains($submenu->url, '/permissions')) {
            $isPermissionsMenu = true;
          }
          if (isset($submenu->slug) && str_contains($submenu->slug, 'permissions')) {
            $isPermissionsMenu = true;
          }

          // Only show permissions menu if user is admin
          if ($isPermissionsMenu) {
            return $isAdmin;
          }
          return true;
        });
        // Re-index array after filtering
        $menu->submenu = array_values($menu->submenu);
      }
      return true;
    });

    $menuData->menu = array_values($filteredMenu);
    return $menuData;
  }
}
