<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, HasApiTokens, Notifiable, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role_id',
        'shop_id',
        'approved',
        'locked',
        'phone',
        'profile_image',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'approved' => 'boolean',
            'locked' => 'boolean',
        ];
    }

    public function role()
    {
        return $this->belongsTo(Role::class);
    }

    public function shop()
    {
        return $this->belongsTo(Shop::class);
    }

    public function rentals()
    {
        return $this->hasMany(Rental::class, 'rented_by');
    }

    public function stockMovements()
    {
        return $this->hasMany(StockMovement::class, 'created_by');
    }

    public function payments()
    {
        return $this->hasMany(Payment::class, 'paid_by');
    }

    public function returns()
    {
        return $this->hasMany(RentalReturn::class, 'returned_by');
    }

    public function documents()
    {
        return $this->morphMany(Document::class, 'model');
    }

    public function permissions()
    {
        return $this->belongsToMany(Permission::class, 'user_permissions')
            ->withPivot('granted')
            ->withTimestamps();
    }

    public function userPermissions()
    {
        return $this->hasMany(UserPermission::class);
    }

    public function isAdmin()
    {
        return $this->role_id === 1; // Assuming admin role_id is 1
    }

    public function isShopManager()
    {
        return $this->role_id === 2; // Assuming shop_manager role_id is 2
    }

    public function canLogin()
    {
        return $this->approved && !$this->locked;
    }

    /**
     * Check if user has a specific permission
     * Priority: User-specific permissions override role permissions
     */
    public function hasPermission($permissionName)
    {
        // Admin has all permissions
        if ($this->isAdmin()) {
            return true;
        }

        $permission = Permission::where('name', $permissionName)->first();
        if (!$permission) {
            return false;
        }

        // Check user-specific permissions first (these override role permissions)
        $userPermission = $this->userPermissions()
            ->where('permission_id', $permission->id)
            ->first();

        // If user has explicit permission set, use that (even if false - explicit deny)
        if ($userPermission) {
            return $userPermission->granted === true;
        }

        // No user-specific override - check role-based permissions (default)
        if ($this->role) {
            $rolePermission = \App\Models\RolePermission::where('role_id', $this->role_id)
                ->where('permission_id', $permission->id)
                ->where('granted', true)
                ->first();

            return $rolePermission !== null;
        }

        // No role, no user permission - deny by default
        return false;
    }

    /**
     * Get all granted permission names for the user
     */
    public function getPermissionNames()
    {
        if ($this->isAdmin()) {
            return Permission::pluck('name')->toArray();
        }

        $userPermissionIds = $this->userPermissions()
            ->where('granted', true)
            ->pluck('permission_id');

        $rolePermissionIds = $this->role
            ? \App\Models\RolePermission::where('role_id', $this->role_id)
                ->where('granted', true)
                ->pluck('permission_id')
            : collect();

        $allPermissionIds = $userPermissionIds->merge($rolePermissionIds)->unique();

        return Permission::whereIn('id', $allPermissionIds)->pluck('name')->toArray();
    }
}
