<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
  public function index()
  {
    $users = User::with(['role', 'shop'])->paginate(10);
    $roles = Role::all();
    $shops = Shop::all();
    return view('module.user.index', compact('users', 'roles', 'shops'));
  }

  public function create()
  {
    $roles = Role::all();
    $shops = Shop::all();
    return view('module.user.create', compact('roles', 'shops'));
  }

  public function store(Request $request)
  {
    $request->validate([
      'name' => 'required|string|max:255',
      'email' => 'required|email|unique:users',
      'password' => 'required|string|min:8',
      'role_id' => 'required|exists:roles,id',
      'shop_id' => 'nullable|exists:shops,id',
      'phone' => 'nullable|string',
      'approved' => 'boolean',
      'locked' => 'boolean',
    ]);

    $user = User::create([
      'name' => $request->name,
      'email' => $request->email,
      'password' => Hash::make($request->password),
      'role_id' => $request->role_id,
      'shop_id' => $request->shop_id,
      'phone' => $request->phone,
      'approved' => $request->approved ?? false,
      'locked' => $request->locked ?? false,
    ]);

    return redirect()->route('users.index')->with('success', 'User created successfully');
  }

  public function update(Request $request, User $user)
  {
    $request->validate([
      'name' => 'required|string|max:255',
      'email' => 'required|email|unique:users,email,' . $user->id,
      'password' => 'nullable|string|min:8',
      'role_id' => 'required|exists:roles,id',
      'shop_id' => 'nullable|exists:shops,id',
      'phone' => 'nullable|string',
      'approved' => 'boolean',
      'locked' => 'boolean',
    ]);

    $data = [
      'name' => $request->name,
      'email' => $request->email,
      'role_id' => $request->role_id,
      'shop_id' => $request->shop_id,
      'phone' => $request->phone,
      'approved' => $request->approved ?? false,
      'locked' => $request->locked ?? false,
    ];

    if ($request->filled('password')) {
      $data['password'] = Hash::make($request->password);
    }

    $user->update($data);

    return redirect()->route('users.index')
      ->with('success', 'User updated successfully');
  }

  public function destroy(User $user)
  {
    $user->delete();

    return redirect()->route('users.index')
      ->with('success', 'User deleted successfully');
  }
}
