<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class SettingsController extends Controller
{
    /**
     * Display account settings page
     */
    public function account()
    {
        $user = Auth::user();
        return view('module.settings.account', compact('user'));
    }

    /**
     * Update account settings
     */
    public function updateAccount(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'firstName' => 'nullable|string|max:255',
            'lastName' => 'nullable|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'organization' => 'nullable|string|max:255',
            'phoneNumber' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'state' => 'nullable|string|max:255',
            'zipCode' => 'nullable|string|max:10',
            'country' => 'nullable|string|max:255',
            'language' => 'nullable|string|max:50',
            'timezone' => 'nullable|string|max:100',
            'currency' => 'nullable|string|max:10',
            'upload' => 'nullable|image|mimes:jpg,jpeg,png|max:800',
        ]);

        // Update user name (combine first and last name if provided)
        if (isset($validated['firstName']) || isset($validated['lastName'])) {
            $name = trim(($validated['firstName'] ?? '') . ' ' . ($validated['lastName'] ?? ''));
            if ($name) {
                $user->name = $name;
            }
        }

        // Update email
        $user->email = $validated['email'];

        // Update phone if provided
        if (isset($validated['phoneNumber'])) {
            $user->phone = $validated['phoneNumber'];
        }

        // Handle profile image upload
        if ($request->hasFile('upload')) {
            // Delete old profile image if exists
            if ($user->profile_image && Storage::disk('public')->exists($user->profile_image)) {
                Storage::disk('public')->delete($user->profile_image);
            }
            
            $image = $request->file('upload');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $imagePath = $image->storeAs('profiles/users', $imageName, 'public');
            $user->profile_image = $imagePath;
        }

        $user->save();

        // Store other settings in session or user preferences table if needed
        // For now, we'll just update the basic user fields

        return redirect()->route('settings.account')
            ->with('success', 'Account settings updated successfully.');
    }
}
