<?php

namespace App\Http\Controllers;

use App\Models\Permission;
use App\Models\User;
use App\Models\UserPermission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PermissionController extends Controller
{
    /**
     * Display the permissions management page
     */
    public function index()
    {
        $users = User::all();
        $permissions = Permission::orderBy('module')->orderBy('action')->get();
        
        // Group permissions by module
        $groupedPermissions = $permissions->groupBy('module');
        
        return view('module.permissions.index', compact('users', 'permissions', 'groupedPermissions'));
    }

    /**
     * Get permissions for a specific user
     */
    public function getUserPermissions($userId)
    {
        $user = User::findOrFail($userId);
        
        // Get all permissions
        $allPermissions = Permission::orderBy('module')->orderBy('action')->get();
        
        // Get user's explicit permissions (user-specific overrides)
        $userPermissions = UserPermission::where('user_id', $userId)
            ->get()
            ->keyBy('permission_id');
        
        // Get role-based permissions (default permissions for the role)
        $rolePermissionIds = [];
        if ($user->role) {
            $rolePermissionIds = \App\Models\RolePermission::where('role_id', $user->role_id)
                ->where('granted', true)
                ->pluck('permission_id')
                ->toArray();
        }
        
        $permissions = [];
        foreach ($allPermissions as $permission) {
            // Check if user has explicit permission override
            $userPermission = $userPermissions->get($permission->id);
            
            // If user has explicit permission, use that (user-specific override)
            // Otherwise, use role-based permission (default)
            $granted = false;
            if ($userPermission) {
                // User has explicit permission set - use it
                $granted = $userPermission->granted;
            } else {
                // No user-specific override - use role default
                $granted = in_array($permission->id, $rolePermissionIds);
            }
            
            $permissions[] = [
                'id' => $permission->id,
                'name' => $permission->name,
                'module' => $permission->module,
                'action' => $permission->action,
                'description' => $permission->description,
                'granted' => $granted,
                'from_role' => $userPermission === null, // true if permission comes from role, false if user-specific
            ];
        }
        
        return response()->json([
            'success' => true,
            'permissions' => $permissions,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => $user->role ? $user->role->name : null,
            ]
        ]);
    }

    /**
     * Update user permissions
     * Note: Only user-specific overrides are stored. If a permission is not set, it falls back to role permissions.
     */
    public function updateUserPermissions(Request $request, $userId)
    {
        $request->validate([
            'permissions' => 'required|array',
            'permissions.*' => 'required|boolean',
        ]);

        DB::beginTransaction();
        try {
            $user = User::findOrFail($userId);
            
            // Admin cannot have permissions modified
            if ($user->isAdmin()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Admin users have all permissions by default and cannot be modified.'
                ], 403);
            }

            $permissionIds = array_keys($request->permissions);
            
            // Get all valid permission IDs
            $validPermissionIds = Permission::whereIn('id', $permissionIds)->pluck('id')->toArray();
            
            // Get role permissions for comparison
            $rolePermissionIds = [];
            if ($user->role) {
                $rolePermissionIds = \App\Models\RolePermission::where('role_id', $user->role_id)
                    ->where('granted', true)
                    ->pluck('permission_id')
                    ->toArray();
            }
            
            // Process each permission
            foreach ($validPermissionIds as $permissionId) {
                $requestedGranted = $request->permissions[$permissionId] === true || $request->permissions[$permissionId] === 'true';
                $roleHasPermission = in_array($permissionId, $rolePermissionIds);
                
                // If user permission matches role permission, remove user override (use role default)
                if ($requestedGranted === $roleHasPermission) {
                    // Remove user-specific override - will use role default
                    UserPermission::where('user_id', $userId)
                        ->where('permission_id', $permissionId)
                        ->delete();
                } else {
                    // User wants different permission than role - create/update user override
                    UserPermission::updateOrCreate(
                        [
                            'user_id' => $userId,
                            'permission_id' => $permissionId,
                        ],
                        [
                            'granted' => $requestedGranted,
                        ]
                    );
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Permissions updated successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating permissions: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle a single permission for a user
     * If permission matches role default, removes user override
     */
    public function togglePermission(Request $request, $userId)
    {
        $request->validate([
            'permission_id' => 'required|exists:permissions,id',
            'granted' => 'required|boolean',
        ]);

        $user = User::findOrFail($userId);
        
        // Admin cannot have permissions modified
        if ($user->isAdmin()) {
            return response()->json([
                'success' => false,
                'message' => 'Admin users have all permissions by default and cannot be modified.'
            ], 403);
        }

        // Get role permission for this permission
        $roleHasPermission = false;
        if ($user->role) {
            $rolePermission = \App\Models\RolePermission::where('role_id', $user->role_id)
                ->where('permission_id', $request->permission_id)
                ->where('granted', true)
                ->first();
            $roleHasPermission = $rolePermission !== null;
        }

        $requestedGranted = $request->granted === true || $request->granted === 'true';

        // If user permission matches role permission, remove user override (use role default)
        if ($requestedGranted === $roleHasPermission) {
            UserPermission::where('user_id', $userId)
                ->where('permission_id', $request->permission_id)
                ->delete();
        } else {
            // User wants different permission than role - create/update user override
            UserPermission::updateOrCreate(
                [
                    'user_id' => $userId,
                    'permission_id' => $request->permission_id,
                ],
                [
                    'granted' => $requestedGranted,
                ]
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Permission updated successfully'
        ]);
    }
}
