<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Rental;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    public function index()
    {
        $invoices = Invoice::with(['rental.shop', 'rental.customer'])->paginate(10);
        return view('module.invoice.index', compact('invoices'));
    }

    public function generate(Rental $rental)
    {
        // Check if agreement exists (auto-create if missing for older rentals)
        $agreement = $rental->agreement;
        
        if (!$agreement) {
            // Auto-create agreement if it doesn't exist
            $agreement = \App\Models\RentalAgreement::create([
                'rental_id' => $rental->id,
                'agreement_number' => 'AGR-' . $rental->id,
                'issued_at' => $rental->rental_date ?? now(),
                'status' => $rental->status === 'returned' ? 'equipment_handed_over' : 'pending',
            ]);
        }

        // Allow invoice generation if:
        // 1. Rental status is "returned" (equipment is back), OR
        // 2. Agreement status is "equipment_handed_over"
        $canGenerate = $rental->status === 'returned' || 
                      ($agreement && $agreement->canGenerateInvoice());

        if (!$canGenerate) {
            return redirect()->back()->with('error', 'Cannot generate invoice. Equipment must be returned or handed over first.');
        }

        // Check if invoice exists
        $invoice = Invoice::where('rental_id', $rental->id)->first();

        if (!$invoice) {
            // Create invoice with same identifier as agreement (rental_id) for easy tracking
            $invoice = \App\Models\Invoice::create([
                'rental_id' => $rental->id,
                'invoice_number' => 'INV-' . $rental->id,
                'issued_at' => now(),
                'due_date' => $rental->expected_return_date ?? now()->addDays(7),
                'amount' => $rental->grand_total,
                'status' => 'unpaid',
            ]);
        }

        return redirect()->route('invoices.preview', $invoice);
    }

    public function edit(Invoice $invoice)
    {
        // Check if invoice can be edited (no payments processed)
        if (!$invoice->canBeEdited()) {
            return redirect()->route('invoices.show', $invoice)
                ->with('error', 'Cannot edit invoice. Payment has already been processed.');
        }

        $invoice->load(['rental.shop', 'rental.customer', 'rental.rentalItems.equipment', 'rental.rentedBy', 'rental.agreement', 'payments']);
        return view('module.invoice.edit', compact('invoice'));
    }

    public function update(Request $request, Invoice $invoice)
    {
        // Check if invoice can be edited (no payments processed)
        if (!$invoice->canBeEdited()) {
            return redirect()->route('invoices.show', $invoice)
                ->with('error', 'Cannot edit invoice. Payment has already been processed.');
        }

        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'due_date' => 'nullable|date',
        ]);

        $invoice->update($validated);

        return redirect()->route('invoices.show', $invoice)
            ->with('success', 'Invoice updated successfully.');
    }

    public function show(Invoice $invoice)
    {
        $invoice->load(['rental.shop', 'rental.customer', 'rental.rentalItems.equipment', 'rental.rentedBy', 'rental.agreement', 'payments']);
        return view('module.invoice.show', compact('invoice'));
    }

    public function preview(Invoice $invoice)
    {
        $invoice->load(['rental.shop', 'rental.customer', 'rental.rentalItems.equipment', 'rental.rentedBy', 'rental.agreement', 'payments']);
        return view('module.invoice.preview', compact('invoice'));
    }

    public function download(Invoice $invoice)
    {
        $invoice->load(['rental.shop', 'rental.customer', 'rental.rentalItems.equipment', 'rental.rentedBy']);

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('invoices.pdf', compact('invoice'));

        // Save PDF
        $pdfPath = 'invoices/' . $invoice->invoice_number . '.pdf';
        \Storage::disk('public')->put($pdfPath, $pdf->output());
        $invoice->update(['pdf_path' => $pdfPath]);

        return $pdf->download($invoice->invoice_number . '.pdf');
    }

    public function thermal(Invoice $invoice)
    {
        $invoice->load(['rental.shop', 'rental.customer', 'rental.rentalItems.equipment', 'rental.rentedBy', 'payments']);
        return view('invoices.thermal', compact('invoice'));
    }
}
