<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Rental;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\Shop;
use App\Models\Stock;
use App\Models\Equipment;
use App\Models\Customer;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
  public function index()
  {
    $user = auth()->user();
    $isAdmin = $user->isAdmin();
    $shopId = $user->shop_id;

    // Date ranges
    $today = Carbon::today();
    $thisMonth = Carbon::now()->startOfMonth();
    $lastMonth = Carbon::now()->subMonth()->startOfMonth();
    $lastMonthEnd = Carbon::now()->subMonth()->endOfMonth();

    // Base queries - filter by shop if shop manager
    $rentalQuery = Rental::query();
    $invoiceQuery = Invoice::query();
    $paymentQuery = Payment::query();

    if (!$isAdmin && $shopId) {
      $rentalQuery->where('shop_id', $shopId);
      $invoiceQuery->whereHas('rental', function($q) use ($shopId) {
        $q->where('shop_id', $shopId);
      });
      $paymentQuery->whereHas('rental', function($q) use ($shopId) {
        $q->where('shop_id', $shopId);
      });
    }

    // Rental Statistics
    $totalRentals = (clone $rentalQuery)->count();
    $activeRentals = (clone $rentalQuery)->where('status', 'active')->count();
    $returnedRentals = (clone $rentalQuery)->where('status', 'returned')->count();
    $todayRentals = (clone $rentalQuery)->whereDate('rental_date', $today)->count();
    $thisMonthRentals = (clone $rentalQuery)->whereDate('rental_date', '>=', $thisMonth)->count();
    $lastMonthRentals = (clone $rentalQuery)
      ->whereDate('rental_date', '>=', $lastMonth)
      ->whereDate('rental_date', '<=', $lastMonthEnd)
      ->count();

    // Revenue Statistics
    $totalRevenue = (clone $rentalQuery)
      ->where('status', '!=', 'cancelled')
      ->sum('grand_total') ?? 0;

    $todayRevenue = (clone $rentalQuery)
      ->where('status', '!=', 'cancelled')
      ->whereDate('rental_date', $today)
      ->sum('grand_total') ?? 0;

    $thisMonthRevenue = (clone $rentalQuery)
      ->where('status', '!=', 'cancelled')
      ->whereDate('rental_date', '>=', $thisMonth)
      ->sum('grand_total') ?? 0;

    $lastMonthRevenue = (clone $rentalQuery)
      ->where('status', '!=', 'cancelled')
      ->whereDate('rental_date', '>=', $lastMonth)
      ->whereDate('rental_date', '<=', $lastMonthEnd)
      ->sum('grand_total') ?? 0;

    // Payment Statistics
    $totalPaid = (clone $paymentQuery)->sum('amount') ?? 0;
    $unpaidInvoices = (clone $invoiceQuery)
      ->where('status', '!=', 'paid')
      ->count();
    $unpaidAmount = (clone $invoiceQuery)
      ->where('status', '!=', 'paid')
      ->sum('amount') ?? 0;

    // Stock Alerts
    $stockQuery = Stock::query();
    if (!$isAdmin && $shopId) {
      $stockQuery->where('shop_id', $shopId);
    }

    $lowStockItems = (clone $stockQuery)
      ->with(['equipment', 'shop'])
      ->whereRaw('(quantity - reserved) <= minimum_threshold')
      ->where('minimum_threshold', '>', 0)
      ->get();

    $outOfStockItems = (clone $stockQuery)
      ->with(['equipment', 'shop'])
      ->whereRaw('(quantity - reserved) <= 0')
      ->get();

    // Recent Rentals
    $recentRentals = (clone $rentalQuery)
      ->with(['shop', 'customer', 'rentedBy'])
      ->orderBy('created_at', 'desc')
      ->limit(10)
      ->get();

    // Shop Performance (only for admin)
    $shopPerformance = [];
    if ($isAdmin) {
      $shopPerformance = Shop::withCount(['rentals' => function($query) use ($thisMonth) {
        $query->where('rental_date', '>=', $thisMonth)
              ->where('status', '!=', 'cancelled');
      }])
      ->withSum(['rentals' => function($query) use ($thisMonth) {
        $query->where('rental_date', '>=', $thisMonth)
              ->where('status', '!=', 'cancelled');
      }], 'grand_total')
      ->get();
    }

    // Monthly Revenue Chart Data (last 6 months)
    $monthlyRevenue = [];
    for ($i = 5; $i >= 0; $i--) {
      $monthStart = Carbon::now()->subMonths($i)->startOfMonth();
      $monthEnd = Carbon::now()->subMonths($i)->endOfMonth();

      $monthQuery = (clone $rentalQuery)
        ->where('status', '!=', 'cancelled')
        ->whereDate('rental_date', '>=', $monthStart)
        ->whereDate('rental_date', '<=', $monthEnd);

      $monthlyRevenue[] = [
        'month' => $monthStart->format('M Y'),
        'revenue' => $monthQuery->sum('grand_total') ?? 0,
        'count' => $monthQuery->count(),
      ];
    }

    // Equipment Utilization (top 5 most rented)
    $equipmentUtilization = DB::table('rental_items')
      ->join('rentals', 'rental_items.rental_id', '=', 'rentals.id')
      ->join('equipments', 'rental_items.equipment_id', '=', 'equipments.id')
      ->where('rentals.status', '!=', 'cancelled')
      ->when(!$isAdmin && $shopId, function($query) use ($shopId) {
        return $query->where('rentals.shop_id', $shopId);
      })
      ->select('equipments.id', 'equipments.name', DB::raw('SUM(rental_items.quantity) as total_rented'))
      ->groupBy('equipments.id', 'equipments.name')
      ->orderBy('total_rented', 'desc')
      ->limit(5)
      ->get();

    // Status Distribution
    $statusDistribution = (clone $rentalQuery)
      ->select('status', DB::raw('COUNT(*) as count'))
      ->groupBy('status')
      ->get()
      ->pluck('count', 'status');

    // Daily Rental Trends (last 14 days)
    $dailyRentals = [];
    for ($i = 13; $i >= 0; $i--) {
      $date = Carbon::now()->subDays($i);
      $dateStart = $date->copy()->startOfDay();
      $dateEnd = $date->copy()->endOfDay();

      $dayQuery = (clone $rentalQuery)
        ->whereDate('rental_date', '>=', $dateStart)
        ->whereDate('rental_date', '<=', $dateEnd);

      $dayRevenueQuery = (clone $rentalQuery)
        ->where('status', '!=', 'cancelled')
        ->whereDate('rental_date', '>=', $dateStart)
        ->whereDate('rental_date', '<=', $dateEnd);

      $dailyRentals[] = [
        'date' => $date->format('M d'),
        'day' => $date->format('D'),
        'count' => $dayQuery->count(),
        'revenue' => $dayRevenueQuery->sum('grand_total') ?? 0,
      ];
    }

    // Total counts
    $totalShops = $isAdmin ? Shop::count() : 1;
    $totalEquipments = Equipment::count();
    $totalCustomers = Customer::count();

    return view('module.dashboard.index', compact(
      'totalRentals',
      'activeRentals',
      'returnedRentals',
      'todayRentals',
      'thisMonthRentals',
      'lastMonthRentals',
      'totalRevenue',
      'todayRevenue',
      'thisMonthRevenue',
      'lastMonthRevenue',
      'totalPaid',
      'unpaidInvoices',
      'unpaidAmount',
      'lowStockItems',
      'outOfStockItems',
      'recentRentals',
      'shopPerformance',
      'monthlyRevenue',
      'equipmentUtilization',
      'statusDistribution',
      'dailyRentals',
      'totalShops',
      'totalEquipments',
      'totalCustomers',
      'isAdmin'
    ));
  }
}
