<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class UserController extends Controller
{
  // List users with pagination
  public function index()
  {
    $users = User::with(['role', 'shop'])->paginate(10);

    return response()->json([
      'success' => true,
      'users' => $users
    ]);
  }

  // Store new user
  public function store(Request $request)
  {
    $validated = $request->validate([
      'name' => 'required|string|max:255',
      'email' => 'required|email|unique:users',
      'password' => 'required|string|min:8',
      'role_id' => 'required|exists:roles,id',
      'shop_id' => 'nullable|exists:shops,id',
      'phone' => 'nullable|string',
      'approved' => 'boolean',
      'locked' => 'boolean',
      'profile_image' => 'nullable|image|mimes:jpg,jpeg,png|max:1024',
    ]);

    $data = [
      'name' => $validated['name'],
      'email' => $validated['email'],
      'password' => Hash::make($validated['password']),
      'role_id' => $validated['role_id'],
      'shop_id' => $validated['shop_id'] ?? null,
      'phone' => $validated['phone'] ?? null,
      'approved' => $validated['approved'] ?? false,
      'locked' => $validated['locked'] ?? false,
    ];

    // Handle profile image upload
    if ($request->hasFile('profile_image')) {
      $image = $request->file('profile_image');
      $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
      $imagePath = $image->storeAs('profiles/users', $imageName, 'public');
      $data['profile_image'] = $imagePath;
    }

    $user = User::create($data);

    return response()->json([
      'success' => true,
      'message' => 'User created successfully',
      'data' => $user->load(['role', 'shop'])
    ]);
  }

  // Update existing user
  public function update(Request $request, User $user)
  {
    $validated = $request->validate([
      'name' => 'required|string|max:255',
      'email' => 'required|email|unique:users,email,' . $user->id,
      'password' => 'nullable|string|min:8',
      'role_id' => 'required|exists:roles,id',
      'shop_id' => 'nullable|exists:shops,id',
      'phone' => 'nullable|string',
      'approved' => 'nullable|boolean',
      'locked' => 'nullable|boolean',
      'profile_image' => 'nullable|image|mimes:jpg,jpeg,png|max:1024',
    ]);

    $data = [
      'name' => $validated['name'],
      'email' => $validated['email'],
      'role_id' => $validated['role_id'],
      'shop_id' => $validated['shop_id'] ?? null,
      'phone' => $validated['phone'] ?? null,
      'approved' => isset($validated['approved']) ? filter_var($validated['approved'], FILTER_VALIDATE_BOOLEAN) : $user->approved,
      'locked' => isset($validated['locked']) ? filter_var($validated['locked'], FILTER_VALIDATE_BOOLEAN) : $user->locked,
    ];

    if (isset($validated['password']) && !empty($validated['password'])) {
      $data['password'] = Hash::make($validated['password']);
    }

    // Handle profile image upload
    if ($request->hasFile('profile_image')) {
      // Delete old profile image if exists
      if ($user->profile_image && Storage::disk('public')->exists($user->profile_image)) {
        Storage::disk('public')->delete($user->profile_image);
      }
      
      $image = $request->file('profile_image');
      $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
      $imagePath = $image->storeAs('profiles/users', $imageName, 'public');
      $data['profile_image'] = $imagePath;
    }

    $user->update($data);

    return response()->json([
      'success' => true,
      'message' => 'User updated successfully',
      'data' => $user->load(['role', 'shop'])
    ]);
  }

  // Delete a user
  public function destroy(User $user)
  {
    $user->delete();

    return response()->json([
      'success' => true,
      'message' => 'User deleted successfully'
    ]);
  }

  // Optional: show a single user
  public function show(Request $request, User $user)
  {
    $with = $request->get('with', '');
    $relations = ['role', 'shop'];
    
    if (strpos($with, 'documents') !== false) {
      $relations[] = 'documents';
    }
    
    return response()->json([
      'success' => true,
      'user' => $user->load($relations)
    ]);
  }
}
