<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Stock;
use Illuminate\Http\Request;

class StockController extends Controller
{
    public function index(Request $request)
    {
        $query = Stock::with(['shop', 'equipment.category']);

        if ($request->has('shop_id')) {
            $query->where('shop_id', $request->shop_id);
        }

        // If all parameter is set, return all stocks without pagination
        if ($request->has('all') && $request->get('all') == 'true') {
            $stocks = $query->get();
            return response()->json([
                'success' => true,
                'data' => $stocks
            ]);
        }

        $stocks = $query->paginate(10);
        return response()->json([
            'success' => true,
            'data' => $stocks
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'shop_id' => 'required|exists:shops,id',
            'equipment_id' => 'required|exists:equipments,id',
            'quantity' => 'required|integer|min:0',
            'reserved' => 'nullable|integer|min:0',
            'minimum_threshold' => 'nullable|integer|min:0',
        ]);

        $stock = Stock::updateOrCreate(
            ['shop_id' => $validated['shop_id'], 'equipment_id' => $validated['equipment_id']],
            $validated
        );

        return response()->json([
            'success' => true,
            'message' => 'Stock created/updated successfully',
            'data' => $stock->load(['shop', 'equipment'])
        ]);
    }

    public function update(Request $request, Stock $stock)
    {
        $validated = $request->validate([
            'quantity' => 'required|integer|min:0',
            'reserved' => 'nullable|integer|min:0',
            'minimum_threshold' => 'nullable|integer|min:0',
        ]);

        $stock->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Stock updated successfully',
            'data' => $stock->load(['shop', 'equipment'])
        ]);
    }

    public function destroy(Stock $stock)
    {
        $stock->delete();

        return response()->json([
            'success' => true,
            'message' => 'Stock deleted successfully'
        ]);
    }

    public function show(Stock $stock)
    {
        return response()->json([
            'success' => true,
            'data' => $stock->load(['shop', 'equipment'])
        ]);
    }
}
