<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\RentalItem;
use App\Models\Rental;
use App\Models\Equipment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RentalItemController extends Controller
{
    public function index($rentalId)
    {
        $rental = Rental::findOrFail($rentalId);
        $items = $rental->rentalItems()->with('equipment')->get();

        return response()->json([
            'success' => true,
            'data' => $items
        ]);
    }

    public function store(Request $request, $rentalId)
    {
        $rental = Rental::findOrFail($rentalId);

        $validated = $request->validate([
            'equipment_id' => 'required|exists:equipments,id',
            'quantity' => 'required|integer|min:1',
            'days' => 'required|numeric|min:0.5',
            'serial_numbers' => 'nullable|array',
        ]);

        $equipment = Equipment::findOrFail($validated['equipment_id']);

        // Check stock availability
        $stock = \App\Models\Stock::where('shop_id', $rental->shop_id)
            ->where('equipment_id', $equipment->id)
            ->first();

        $available = ($stock?->quantity ?? 0) - ($stock?->reserved ?? 0);

        if ($available < $validated['quantity']) {
            return response()->json([
                'success' => false,
                'message' => "Insufficient stock. Available: {$available}, Requested: {$validated['quantity']}"
            ], 400);
        }

        DB::beginTransaction();
        try {
            // Calculate subtotal
            $subtotal = $equipment->daily_rate * $validated['quantity'] * $validated['days'];

            // Create rental item
            $rentalItem = RentalItem::create([
                'rental_id' => $rental->id,
                'equipment_id' => $equipment->id,
                'quantity' => $validated['quantity'],
                'daily_rate' => $equipment->daily_rate,
                'days' => $validated['days'],
                'subtotal' => $subtotal,
                'serial_numbers' => $validated['serial_numbers'] ?? null,
            ]);

            // Update stock if rental is active
            if ($rental->status === 'active') {
                if ($stock) {
                    $stock->increment('reserved', $validated['quantity']);
                }
            }

            // Update rental totals
            $rental->refresh();
            $totalAmount = $rental->rentalItems()->sum('subtotal');
            $rental->update([
                'total_amount' => $totalAmount,
                'grand_total' => $totalAmount - ($rental->discount ?? 0)
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Rental item added successfully',
                'data' => $rentalItem->load('equipment')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error adding rental item: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $rentalId, RentalItem $rentalItem)
    {
        $rental = Rental::findOrFail($rentalId);

        $validated = $request->validate([
            'quantity' => 'sometimes|integer|min:1',
            'days' => 'sometimes|numeric|min:0.5',
            'serial_numbers' => 'nullable|array',
        ]);

        DB::beginTransaction();
        try {
            $oldQuantity = $rentalItem->quantity;
            $newQuantity = $validated['quantity'] ?? $oldQuantity;
            $newDays = $validated['days'] ?? $rentalItem->days;

            // Update rental item
            $subtotal = $rentalItem->daily_rate * $newQuantity * $newDays;
            $rentalItem->update([
                'quantity' => $newQuantity,
                'days' => $newDays,
                'subtotal' => $subtotal,
                'serial_numbers' => $validated['serial_numbers'] ?? $rentalItem->serial_numbers,
            ]);

            // Update stock if rental is active
            if ($rental->status === 'active' && $oldQuantity != $newQuantity) {
                $stock = \App\Models\Stock::where('shop_id', $rental->shop_id)
                    ->where('equipment_id', $rentalItem->equipment_id)
                    ->first();

                if ($stock) {
                    $difference = $newQuantity - $oldQuantity;
                    $stock->increment('reserved', $difference);
                }
            }

            // Update rental totals
            $totalAmount = $rental->rentalItems()->sum('subtotal');
            $rental->update([
                'total_amount' => $totalAmount,
                'grand_total' => $totalAmount - ($rental->discount ?? 0)
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Rental item updated successfully',
                'data' => $rentalItem->load('equipment')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating rental item: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy($rentalId, RentalItem $rentalItem)
    {
        $rental = Rental::findOrFail($rentalId);

        DB::beginTransaction();
        try {
            // Update stock if rental is active
            if ($rental->status === 'active') {
                $stock = \App\Models\Stock::where('shop_id', $rental->shop_id)
                    ->where('equipment_id', $rentalItem->equipment_id)
                    ->first();

                if ($stock) {
                    $stock->decrement('reserved', $rentalItem->quantity);
                }
            }

            $rentalItem->delete();

            // Update rental totals
            $totalAmount = $rental->rentalItems()->sum('subtotal');
            $rental->update([
                'total_amount' => $totalAmount,
                'grand_total' => $totalAmount - ($rental->discount ?? 0)
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Rental item deleted successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error deleting rental item: ' . $e->getMessage()
            ], 500);
        }
    }
}
