<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Rental;
use App\Services\StockService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RentalController extends Controller
{
    public function index()
    {
        $rentals = Rental::with(['shop', 'customer', 'rentedBy', 'rentalItems.equipment'])->paginate(10);
        return response()->json([
            'success' => true,
            'data' => $rentals
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'shop_id' => 'required|exists:shops,id',
            'customer_id' => 'nullable|exists:customers,id',
            'customer_snapshot' => 'nullable|array',
            'rented_by' => 'required|exists:users,id',
            'status' => 'required|in:draft,active,returned,cancelled',
            'rental_date' => 'required|date',
            'expected_return_date' => 'nullable|date',
            'remarks' => 'nullable|string',
            'guarantee_type' => 'nullable|in:cash,nic,driving_license,other',
            'guarantee_details' => 'nullable|string|required_if:guarantee_type,other',
            'discount' => 'nullable|numeric|min:0',
            'items' => 'required|array|min:1',
            'items.*.equipment_id' => 'required|exists:equipments,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.days' => 'required|numeric|min:0.5',
            'items.*.serial_numbers' => 'nullable|array',
        ]);

        // Enforce shop restrictions for non-admin users
        $user = \Illuminate\Support\Facades\Auth::user();
        if ($user && !$user->isAdmin() && $user->shop_id) {
            // Non-admin users can only create rentals for their own shop
            if ($validated['shop_id'] != $user->shop_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You can only create rentals for your assigned shop.'
                ], 403);
            }
        }

        DB::beginTransaction();
        try {
            // Calculate totals
            $totalAmount = 0;
            foreach ($validated['items'] as $item) {
                $equipment = \App\Models\Equipment::findOrFail($item['equipment_id']);
                $subtotal = $equipment->daily_rate * $item['quantity'] * $item['days'];
                $totalAmount += $subtotal;
            }

            $discount = $validated['discount'] ?? 0;
            $discountAmount = $totalAmount * ($discount / 100);
            $grandTotal = $totalAmount - $discountAmount;

            // Create rental
            $rental = Rental::create([
                'shop_id' => $validated['shop_id'],
                'customer_id' => $validated['customer_id'] ?? null,
                'customer_snapshot' => $validated['customer_snapshot'] ?? null,
                'rented_by' => $validated['rented_by'],
                'status' => $validated['status'],
                'rental_date' => $validated['rental_date'],
                'expected_return_date' => $validated['expected_return_date'] ?? null,
                'remarks' => $validated['remarks'] ?? null,
                'guarantee_type' => $validated['guarantee_type'] ?? null,
                'guarantee_details' => $validated['guarantee_type'] === 'other' ? ($validated['guarantee_details'] ?? null) : null,
                'total_amount' => $totalAmount,
                'discount' => $discountAmount,
                'grand_total' => $grandTotal,
            ]);

            // Create rental items
            foreach ($validated['items'] as $item) {
                $equipment = \App\Models\Equipment::findOrFail($item['equipment_id']);
                $subtotal = $equipment->daily_rate * $item['quantity'] * $item['days'];

                \App\Models\RentalItem::create([
                    'rental_id' => $rental->id,
                    'equipment_id' => $item['equipment_id'],
                    'quantity' => $item['quantity'],
                    'daily_rate' => $equipment->daily_rate,
                    'days' => $item['days'],
                    'subtotal' => $subtotal,
                    'serial_numbers' => $item['serial_numbers'] ?? null,
                ]);

                // Check stock availability
                if ($validated['status'] === 'active') {
                    $available = \App\Services\StockService::getAvailable($validated['shop_id'], $item['equipment_id']);
                    if ($available < $item['quantity']) {
                        throw new \Exception("Insufficient stock for {$equipment->name}. Available: {$available}, Requested: {$item['quantity']}");
                    }
                }
            }

            // Reserve stock if status is active
            if ($validated['status'] === 'active') {
                \App\Services\StockService::reserveStock($rental);
            }

            // Create rental agreement (always created when rental is created)
            // Use rental_id as the identifier for easy tracking
            $agreement = \App\Models\RentalAgreement::create([
                'rental_id' => $rental->id,
                'agreement_number' => 'AGR-' . $rental->id,
                'issued_at' => now(),
                'status' => 'pending',
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Rental created successfully',
                'data' => [
                    'rental' => $rental->load(['shop', 'customer', 'rentedBy', 'rentalItems.equipment', 'agreement']),
                    'agreement_id' => $agreement->id
                ]
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error creating rental: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, Rental $rental)
    {
        $validated = $request->validate([
            'shop_id' => 'sometimes|exists:shops,id',
            'customer_id' => 'nullable|exists:customers,id',
            'status' => 'sometimes|in:draft,active,returned,cancelled',
            'rental_date' => 'sometimes|date',
            'expected_return_date' => 'nullable|date',
            'actual_return_date' => 'nullable|date',
            'remarks' => 'nullable|string',
            'guarantee_type' => 'nullable|in:cash,nic,driving_license,other',
            'guarantee_details' => 'nullable|string|required_if:guarantee_type,other',
            'total_amount' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'grand_total' => 'nullable|numeric|min:0',
        ]);

        // Enforce shop restrictions for non-admin users
        $user = \Illuminate\Support\Facades\Auth::user();
        if ($user && !$user->isAdmin() && $user->shop_id) {
            // Non-admin users cannot change shop_id
            if (isset($validated['shop_id']) && $validated['shop_id'] != $user->shop_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You can only update rentals for your assigned shop.'
                ], 403);
            }
            // Ensure rental belongs to user's shop
            if ($rental->shop_id != $user->shop_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to update this rental.'
                ], 403);
            }
        }

        $oldStatus = $rental->status;
        $newStatus = $validated['status'] ?? $oldStatus;

        DB::beginTransaction();
        try {
            // Update basic fields
            if (isset($validated['shop_id'])) {
                // Only allow admins to change shop_id
                if ($user && $user->isAdmin()) {
                    $rental->shop_id = $validated['shop_id'];
                }
            }
            if (isset($validated['customer_id'])) {
                $rental->customer_id = $validated['customer_id'];
            }
            if (isset($validated['rental_date'])) {
                $rental->rental_date = $validated['rental_date'];
            }
            if (isset($validated['expected_return_date'])) {
                $rental->expected_return_date = $validated['expected_return_date'];
            }
            if (isset($validated['remarks'])) {
                $rental->remarks = $validated['remarks'];
            }
            if (isset($validated['guarantee_type'])) {
                $rental->guarantee_type = $validated['guarantee_type'];
            }
            if (isset($validated['guarantee_details'])) {
                // Only set guarantee_details if guarantee_type is 'other'
                if ($validated['guarantee_type'] === 'other') {
                    $rental->guarantee_details = $validated['guarantee_details'];
                } else {
                    $rental->guarantee_details = null;
                }
            } elseif (isset($validated['guarantee_type']) && $validated['guarantee_type'] !== 'other') {
                // Clear guarantee_details if guarantee_type is not 'other'
                $rental->guarantee_details = null;
            }

            // Handle status changes
            if ($oldStatus !== $newStatus) {
                // If changing from draft/active to returned/cancelled, release stock
                if (in_array($oldStatus, ['draft', 'active']) && in_array($newStatus, ['returned', 'cancelled'])) {
                    StockService::releaseStock($rental);
                }

                // If changing to active, reserve stock
                if ($newStatus === 'active' && $oldStatus !== 'active') {
                    StockService::reserveStock($rental);
                }

                // If changing from active back to draft, release stock
                if ($oldStatus === 'active' && $newStatus === 'draft') {
                    StockService::releaseStock($rental);
                }

                // Set actual return date if status is returned
                if ($newStatus === 'returned' && !$rental->actual_return_date) {
                    $rental->actual_return_date = now();
                }

                $rental->status = $newStatus;
            }

            // Only update totals if explicitly provided
            if (isset($validated['total_amount'])) {
                $rental->total_amount = $validated['total_amount'];
            }
            if (isset($validated['discount'])) {
                $rental->discount = $validated['discount'];
            }
            if (isset($validated['grand_total'])) {
                $rental->grand_total = $validated['grand_total'];
            }

            $rental->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Rental updated successfully',
                'data' => $rental->load(['shop', 'customer', 'rentedBy', 'rentalItems.equipment'])
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating rental: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy(Rental $rental)
    {
        $rental->delete();

        return response()->json([
            'success' => true,
            'message' => 'Rental deleted successfully'
        ]);
    }

    public function show(Rental $rental)
    {
        return response()->json([
            'success' => true,
            'data' => $rental->load(['shop', 'customer', 'rentedBy', 'rentalItems.equipment'])
        ]);
    }
}
