<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PaymentController extends Controller
{
    public function store(Request $request)
    {
        $validated = $request->validate([
            'invoice_id' => 'required|exists:invoices,id',
            'amount' => 'required|numeric|min:0.01',
            'method' => 'required|string|in:Cash,Bank Transfer,Debit Card,Credit Card,Paypal,Other',
            'paid_at' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $invoice = Invoice::with('rental')->findOrFail($validated['invoice_id']);
            $rental = $invoice->rental;

            // Create payment
            $payment = Payment::create([
                'invoice_id' => $invoice->id,
                'rental_id' => $rental->id,
                'paid_by' => auth()->id(),
                'amount' => $validated['amount'],
                'method' => $validated['method'],
                'paid_at' => $validated['paid_at'],
                'notes' => $validated['notes'] ?? null,
            ]);

            // Update invoice status based on payments
            $totalPaid = $invoice->payments()->sum('amount');
            if ($totalPaid >= $invoice->amount) {
                $invoice->status = 'paid';
            } elseif ($totalPaid > 0) {
                $invoice->status = 'partially_paid';
            }
            $invoice->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment added successfully',
                'data' => $payment->load(['invoice', 'rental', 'paidBy'])
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error adding payment: ' . $e->getMessage()
            ], 500);
        }
    }

    public function index(Request $request)
    {
        $invoiceId = $request->query('invoice_id');
        
        if ($invoiceId) {
            $payments = Payment::where('invoice_id', $invoiceId)
                ->with(['paidBy'])
                ->orderBy('paid_at', 'desc')
                ->get();
        } else {
            $payments = Payment::with(['invoice', 'rental', 'paidBy'])
                ->orderBy('paid_at', 'desc')
                ->paginate(10);
        }

        return response()->json([
            'success' => true,
            'data' => $payments
        ]);
    }
}





