<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Rental;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class InvoiceController extends Controller
{
    public function index()
    {
        $invoices = Invoice::with(['rental.shop', 'rental.customer'])->paginate(10);
        return response()->json([
            'success' => true,
            'data' => $invoices
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'rental_id' => 'required|exists:rentals,id',
        ]);

        $rental = Rental::with(['rentalItems.equipment', 'shop', 'customer'])->findOrFail($validated['rental_id']);

        // Check if invoice already exists
        $existingInvoice = Invoice::where('rental_id', $rental->id)->first();
        if ($existingInvoice) {
            return response()->json([
                'success' => false,
                'message' => 'Invoice already exists for this rental',
                'data' => $existingInvoice
            ], 400);
        }

        DB::beginTransaction();
        try {
            // Generate invoice number using rental_id for consistency with agreement
            $invoiceNumber = 'INV-' . $rental->id;

            $invoice = Invoice::create([
                'rental_id' => $rental->id,
                'invoice_number' => $invoiceNumber,
                'issued_at' => now(),
                'due_date' => $rental->expected_return_date ?? now()->addDays(7),
                'amount' => $rental->grand_total,
                'status' => 'unpaid',
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice created successfully',
                'data' => $invoice->load(['rental.shop', 'rental.customer'])
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error creating invoice: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show(Invoice $invoice)
    {
        return response()->json([
            'success' => true,
            'data' => $invoice->load(['rental.shop', 'rental.customer', 'rental.rentalItems.equipment', 'payments'])
        ]);
    }

    public function update(Request $request, Invoice $invoice)
    {
        $validated = $request->validate([
            'status' => 'sometimes|in:paid,unpaid,partially_paid',
        ]);

        $invoice->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Invoice updated successfully',
            'data' => $invoice
        ]);
    }

    public function destroy(Invoice $invoice)
    {
        $invoice->delete();

        return response()->json([
            'success' => true,
            'message' => 'Invoice deleted successfully'
        ]);
    }

    public function generatePdf(Invoice $invoice)
    {
        $invoice->load(['rental.shop', 'rental.customer', 'rental.rentalItems.equipment', 'rental.rentedBy']);

        // Generate PDF using dompdf
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('invoices.pdf', compact('invoice'));

        // Save PDF to storage
        $pdfPath = 'invoices/' . $invoice->invoice_number . '.pdf';
        \Storage::disk('public')->put($pdfPath, $pdf->output());

        $invoice->update(['pdf_path' => $pdfPath]);

        return $pdf->download($invoice->invoice_number . '.pdf');
    }
}
