<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Equipment;
use Illuminate\Http\Request;

class EquipmentController extends Controller
{
    public function index(Request $request)
    {
        // If all parameter is set, return all equipment without pagination
        if ($request->has('all') && $request->get('all') == 'true') {
            $equipments = Equipment::with('category')->get();
            return response()->json([
                'success' => true,
                'data' => $equipments
            ]);
        }

        $equipments = Equipment::with('category')->paginate(10);
        return response()->json([
            'success' => true,
            'data' => $equipments
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'sku' => 'nullable|string|unique:equipments,sku',
            'name' => 'required|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'description' => 'nullable|string',
            'daily_rate' => 'required|numeric|min:0',
            'replacement_cost' => 'required|numeric|min:0',
            'is_serial_tracked' => 'boolean',
        ]);

        $equipment = Equipment::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Equipment created successfully',
            'data' => $equipment->load('category')
        ]);
    }

    public function update(Request $request, Equipment $equipment)
    {
        $validated = $request->validate([
            'sku' => 'nullable|string|unique:equipments,sku,' . $equipment->id,
            'name' => 'required|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'description' => 'nullable|string',
            'daily_rate' => 'required|numeric|min:0',
            'replacement_cost' => 'required|numeric|min:0',
            'is_serial_tracked' => 'boolean',
        ]);

        $equipment->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Equipment updated successfully',
            'data' => $equipment->load('category')
        ]);
    }

    public function destroy(Equipment $equipment)
    {
        $equipment->delete();

        return response()->json([
            'success' => true,
            'message' => 'Equipment deleted successfully'
        ]);
    }

    public function show(Equipment $equipment)
    {
        return response()->json([
            'success' => true,
            'data' => $equipment->load('category')
        ]);
    }
}
