<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Document;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class DocumentController extends Controller
{
    public function store(Request $request)
    {
        $validated = $request->validate([
            'model_type' => 'required|string',
            'model_id' => 'required|integer',
            'document_type' => 'nullable|string|max:255',
            'document_name' => 'nullable|string|max:255',
            'file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:1024', // 1MB max
        ]);

        $file = $request->file('file');
        $extension = $file->getClientOriginalExtension();
        $fileName = Str::random(40) . '.' . $extension;
        
        // Determine storage path based on model type
        $modelType = $validated['model_type'];
        // Extract model name from full class name (e.g., "App\Models\User" -> "User")
        $modelName = class_basename($modelType);
        $path = $file->storeAs("documents/{$modelName}", $fileName, 'public');

        // Determine file type
        $fileType = in_array(strtolower($extension), ['jpg', 'jpeg', 'png']) ? 'image' : 'pdf';

        // Create document record
        $document = Document::create([
            'model_type' => $modelType,
            'model_id' => $validated['model_id'],
            'file_path' => $path,
            'file_type' => $fileType,
            'document_type' => $validated['document_type'] ?? null,
            'document_name' => $validated['document_name'] ?? $file->getClientOriginalName(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Document uploaded successfully',
            'data' => $document
        ]);
    }

    public function destroy(Document $document)
    {
        // Delete file from storage
        if (Storage::disk('public')->exists($document->file_path)) {
            Storage::disk('public')->delete($document->file_path);
        }

        $document->delete();

        return response()->json([
            'success' => true,
            'message' => 'Document deleted successfully'
        ]);
    }

    public function show(Document $document)
    {
        $filePath = storage_path('app/public/' . $document->file_path);
        
        if (!file_exists($filePath)) {
            return response()->json([
                'success' => false,
                'message' => 'File not found'
            ], 404);
        }

        // Get file extension to determine MIME type
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $mimeTypes = [
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'pdf' => 'application/pdf',
        ];
        
        $mimeType = $mimeTypes[strtolower($extension)] ?? mime_content_type($filePath) ?? 'application/octet-stream';
        
        return response()->file($filePath, [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'inline; filename="' . ($document->document_name ?? basename($document->file_path)) . '"',
            'Cache-Control' => 'public, max-age=3600'
        ]);
    }
}
