<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function index()
    {
        $users = User::with(['role', 'shop'])->paginate(10);
        $roles = Role::all();
        $shops = Shop::all();
        return view('module.user.index', compact('users', 'roles', 'shops'));
    }

    public function approve(User $user)
    {
        $user->update(['approved' => true]);
        return response()->json([
            'success' => true,
            'message' => 'User approved successfully'
        ]);
    }

    public function lock(User $user)
    {
        $user->update(['locked' => true]);
        return response()->json([
            'success' => true,
            'message' => 'User locked successfully'
        ]);
    }

    public function unlock(User $user)
    {
        $user->update(['locked' => false]);
        return response()->json([
            'success' => true,
            'message' => 'User unlocked successfully'
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
            'role_id' => 'required|exists:roles,id',
            'shop_id' => 'nullable|exists:shops,id',
            'phone' => 'nullable|string|max:20',
            'approved' => 'boolean',
        ]);

        $validated['password'] = Hash::make($validated['password']);
        $validated['approved'] = $validated['approved'] ?? false;

        $user = User::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'User created successfully',
            'data' => $user->load(['role', 'shop'])
        ]);
    }

    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:users,email,' . $user->id,
            'password' => 'sometimes|string|min:8',
            'role_id' => 'sometimes|exists:roles,id',
            'shop_id' => 'nullable|exists:shops,id',
            'phone' => 'nullable|string|max:20',
            'approved' => 'boolean',
            'locked' => 'boolean',
        ]);

        if (isset($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        }

        $user->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'User updated successfully',
            'data' => $user->load(['role', 'shop'])
        ]);
    }

    public function destroy(User $user)
    {
        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully'
        ]);
    }
}
