<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Rental;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\Shop;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function profitLoss(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());
        $shopId = $request->get('shop_id');

        $query = Rental::whereBetween('rental_date', [$startDate, $endDate])
            ->where('status', '!=', 'cancelled');

        if ($shopId) {
            $query->where('shop_id', $shopId);
        }

        $rentals = $query->with(['shop', 'rentalItems.equipment'])->get();

        $totalRevenue = $rentals->sum('grand_total');
        $totalDiscount = $rentals->sum('discount');
        $totalRentals = $rentals->count();

        // Calculate costs (this would need actual cost data, for now using a placeholder)
        $totalCost = 0; // Placeholder - would need procurement costs

        $profit = $totalRevenue - $totalCost;

        return response()->json([
            'success' => true,
            'data' => [
                'period' => [
                    'start' => $startDate,
                    'end' => $endDate,
                ],
                'revenue' => [
                    'total' => $totalRevenue,
                    'discount' => $totalDiscount,
                    'net' => $totalRevenue - $totalDiscount,
                ],
                'costs' => [
                    'total' => $totalCost,
                ],
                'profit' => [
                    'total' => $profit,
                    'margin' => $totalRevenue > 0 ? ($profit / $totalRevenue) * 100 : 0,
                ],
                'rentals' => $totalRentals,
            ]
        ]);
    }

    public function salesByShop(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $sales = Rental::whereBetween('rental_date', [$startDate, $endDate])
            ->where('status', '!=', 'cancelled')
            ->select('shop_id', DB::raw('SUM(grand_total) as total_sales'), DB::raw('COUNT(*) as rental_count'))
            ->groupBy('shop_id')
            ->with('shop')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $sales
        ]);
    }

    public function monthlySummary(Request $request)
    {
        $year = $request->get('year', now()->year);

        $summary = Rental::whereYear('rental_date', $year)
            ->where('status', '!=', 'cancelled')
            ->select(
                DB::raw('MONTH(rental_date) as month'),
                DB::raw('SUM(grand_total) as total_sales'),
                DB::raw('COUNT(*) as rental_count')
            )
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $summary
        ]);
    }

    public function paymentSummary(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $payments = Payment::whereBetween('paid_at', [$startDate, $endDate])
            ->select('method', DB::raw('SUM(amount) as total'))
            ->groupBy('method')
            ->get();

        $totalPaid = Payment::whereBetween('paid_at', [$startDate, $endDate])
            ->sum('amount');

        return response()->json([
            'success' => true,
            'data' => [
                'by_method' => $payments,
                'total' => $totalPaid,
            ]
        ]);
    }
}
